/* Copyright (C) 2003-2008 Eric Hsiao <erichs0608@gmail.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

/* Kernel module implementing a protocol set type as a bitmap */

#include <linux/module.h>
#include <linux/ip.h>
#include <linux/tcp.h>
#include <linux/udp.h>
#include <linux/skbuff.h>
#include <linux/errno.h>
#include <asm/uaccess.h>
#include <asm/bitops.h>
#include <linux/spinlock.h>

#include <net/ip.h>

#include <linux/netfilter_ipv4/ip_set_protocol.h>

extern int protocol_match(const struct sk_buff *skb, const void *matchinfo, const u_int32_t flags);

static int
protocol_utest(struct ip_set *set, const void *data, u_int32_t size)
{
	struct ip_set_protocol *map = set->data;
	const struct ip_set_req_protocol *req = data;
	ip_set_ip_t port = req->ip;

	if (port < map->first_ip || port > map->last_ip)
		return -ERANGE;
	
	DP("set: %s, port: %u", set->name, port);
	return !!test_bit(port - map->first_ip, map->members);	
}

static int
protocol_ktest(struct ip_set *set, const struct sk_buff *skb, const u_int32_t *flags)
{
	struct ip_set_protocol *map = set->data;

#if 0
    if ( flags[0] & IPSET_MATCH_DST )
    {
        /* only check the protocols that requires wan-2-lan block */
    }
#endif

    if ( flags[0] & IPSET_MATCH_REJ )
    {
        /* only check the protocols that requires tcp reset */
        return 0;
    }

	return protocol_match(skb, map->members, flags[0]);
}


static int
protocol_uadd(struct ip_set *set, const void *data, u_int32_t size)
{
	struct ip_set_protocol *map = set->data;
	const struct ip_set_req_protocol *req = data;
	ip_set_ip_t port = req->ip;

	if (port < map->first_ip || port > map->last_ip)
		return -ERANGE;
	if (test_and_set_bit(port - map->first_ip, map->members))
		return -EEXIST;
	
	DP("set: %s, port %u", set->name, port);
	return 0;	
	
}

static int
protocol_kadd(struct ip_set *set, const struct sk_buff *skb, const u_int32_t *flags)
{
	return 0;
}

static int
protocol_udel(struct ip_set *set, const void *data, u_int32_t size)
{
	struct ip_set_protocol *map = set->data;
	const struct ip_set_req_protocol *req = data;
	ip_set_ip_t port = req->ip;

	if (port < map->first_ip || port > map->last_ip)
		return -ERANGE;
	if (!test_and_clear_bit(port - map->first_ip, map->members))
		return -EEXIST;
	
	DP("set: %s, port %u", set->name, port);
	return 0;	
	
}

static int
protocol_kdel(struct ip_set *set, const struct sk_buff *skb, const u_int32_t *flags)
{
	return 0;
}


static inline int
__protocol_create(const struct ip_set_req_protocol_create *req,
		 struct ip_set_protocol *map)
{
	if (req->to - req->from > MAX_RANGE) {
		ip_set_printk("range too big, %d elements (max %d)",
			      req->to - req->from + 1, MAX_RANGE+1);
		return -ENOEXEC;
	}
	return bitmap_bytes(req->from, req->to);
}

BITMAP_CREATE(protocol)
BITMAP_DESTROY(protocol)
BITMAP_FLUSH(protocol)

static inline void
__protocol_list_header(const struct ip_set_protocol *map,
		      struct ip_set_req_protocol_create *header)
{
}

BITMAP_LIST_HEADER(protocol)
BITMAP_LIST_MEMBERS_SIZE(protocol, ip_set_ip_t, (map->last_ip - map->first_ip + 1),
			 test_bit(i, map->members))

static void
protocol_list_members(const struct ip_set *set, void *data, char dont_align)
{
	const struct ip_set_protocol *map = set->data;
	uint32_t i, n = 0;
	ip_set_ip_t *d;
	
	if (dont_align) {
		memcpy(data, map->members, map->size);
		return;
	}
	
	for (i = 0; i < map->last_ip - map->first_ip + 1; i++)
		if (test_bit(i, map->members)) {
			d = data + n * IPSET_ALIGN(sizeof(ip_set_ip_t));
			*d = map->first_ip + i;
			n++;
		}
}

IP_SET_TYPE(protocol, IPSET_DATA_SINGLE)

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Eric Hsiao <erichs0608@gmail.com>");
MODULE_DESCRIPTION("protocol type of IP sets");

REGISTER_MODULE(protocol)
