/*
  dlfix.c

  Copyright:
          Copyright (c) 2002, 2003 SFNT Finland Oy.
  All rights reserved.

  Discrete logarithm predefined parameters.
*/

#include "sshincludes.h"
#ifdef SSHDIST_CRYPT_DL
#include "sshmp.h"
#include "sshcrypt.h"
#include "sshbuffer.h"

/* Structure definition of parameters that are needed to store. */

typedef struct
{
  const char *name;
  const char *p;

  /* If q is NULL, it is assumed to be (p-1)/2 (as an integer) */
  const char *q;
  const char *g;
} SshDLFixedParams;

/* All fixed parameters should satisfy atleast:

   - generate randomly (as much as possible)
   - p = q*c, where p and q are both prime
   - g^q (mod p) = 1

   Those restrictions can be ignored somewhat but then naming should not
   include "ssh". */

#define SSH_DEFAULT_DL_MODP_PARAM_NAME "ssh-dl-modp-group-1024bit-1"

const SshDLFixedParams ssh_dlp_fixed_params[] =
{
  {
    /* 1024 bits with 160 bit order. */
    "ssh-dl-modp-group-dsa-1024bit-1",

     /* p */
    "18870663758990450276373834564151209634609185541696849681710763012"
    "47168050869633674293178701749077483982698046837347550634094699186"
    "01446907583820305901431915822615633146396545907952461810328182170"
    "35078613084684987462347222332164074368740586436373161810202065699"
    "4755732156870258013519880675646961814649650297159",

    /* q */
    "994432737688160994497917820104112624205251325913",

    /* g */
    "13752365807134022918504845143590215341528782407193227118193168220"
    "74331779978192018552914874329836913766220048777129900873815708450"
    "16796174527842910698477887372037694495736629521026242476079522482"
    "50227332682970988562298735692890934535992768521461586958206432475"
    "6777888883265989982517946734947352536810316486901"
  },

  {
    /* 1024 bits DSA style. */
    "ssh-dl-modp-group-1024bit-1",

    /* p */
    "179769313486231590770839156793787453197860296048756011706444"
    "423684197180216158519368947833795864925541502180565485980503"
    "646440548199239100050792877003355816639229553136239076508735"
    "759914822574862575007425302077447712589550957937778424442426"
    "617334727629299387668709205606050270810842907692932019128194"
    "467627007",
    /* q is (p-1)/2 */
    NULL,
    /* g */
     "2" },

#ifdef SSHDIST_CRYPT_DL_IKE_GROUPS
  /* IKE groups. */
  {
    "ietf-ike-grp-modp-768",
    "0x"
    "FFFFFFFF FFFFFFFF C90FDAA2 2168C234 C4C6628B 80DC1CD1"
    "29024E08 8A67CC74 020BBEA6 3B139B22 514A0879 8E3404DD"
    "EF9519B3 CD3A431B 302B0A6D F25F1437 4FE1356D 6D51C245"
    "E485B576 625E7EC6 F44C42E9 A63A3620 FFFFFFFF FFFFFFFF",
    NULL,
    "0x2"
  },
  /* Group common to Secure Shell and IKE. */
  {
    "ietf-ike-grp-modp-1024",
    /* prime */
    "0x"
    "FFFFFFFF FFFFFFFF C90FDAA2 2168C234 C4C6628B 80DC1CD1"
    "29024E08 8A67CC74 020BBEA6 3B139B22 514A0879 8E3404DD"
    "EF9519B3 CD3A431B 302B0A6D F25F1437 4FE1356D 6D51C245"
    "E485B576 625E7EC6 F44C42E9 A637ED6B 0BFF5CB6 F406B7ED"
    "EE386BFB 5A899FA5 AE9F2411 7C4B1FE6 49286651 ECE65381"
    "FFFFFFFF FFFFFFFF",
    /* LPF */
    NULL,
    /* generator */
    "0x2"
  },
  {
    "ietf-ike-grp-modp-1536",
    /* prime */
    "0x"
    "FFFFFFFF FFFFFFFF C90FDAA2 2168C234 C4C6628B 80DC1CD1"
    "29024E08 8A67CC74 020BBEA6 3B139B22 514A0879 8E3404DD"
    "EF9519B3 CD3A431B 302B0A6D F25F1437 4FE1356D 6D51C245"
    "E485B576 625E7EC6 F44C42E9 A637ED6B 0BFF5CB6 F406B7ED"
    "EE386BFB 5A899FA5 AE9F2411 7C4B1FE6 49286651 ECE45B3D"
    "C2007CB8 A163BF05 98DA4836 1C55D39A 69163FA8 FD24CF5F"
    "83655D23 DCA3AD96 1C62F356 208552BB 9ED52907 7096966D"
    "670C354E 4ABC9804 F1746C08 CA237327 FFFFFFFF FFFFFFFF",
    /* LPF */
    NULL,
    /* generator */
    "0x2"
  },
  {
    "ietf-ike-grp-modp-2048",
    /* prime */
    "0x"
    "FFFFFFFFFFFFFFFFC90FDAA22168C234C4C6628B80DC1CD1"
    "29024E088A67CC74020BBEA63B139B22514A08798E3404DD"
    "EF9519B3CD3A431B302B0A6DF25F14374FE1356D6D51C245"
    "E485B576625E7EC6F44C42E9A637ED6B0BFF5CB6F406B7ED"
    "EE386BFB5A899FA5AE9F24117C4B1FE649286651ECE45B3D"
    "C2007CB8A163BF0598DA48361C55D39A69163FA8FD24CF5F"
    "83655D23DCA3AD961C62F356208552BB9ED529077096966D"
    "670C354E4ABC9804F1746C08CA18217C32905E462E36CE3B"
    "E39E772C180E86039B2783A2EC07A28FB5C55DF06F4C52C9"
    "DE2BCBF6955817183995497CEA956AE515D2261898FA0510"
    "15728E5A8AACAA68FFFFFFFFFFFFFFFF",
    /* LPF */
    NULL,
    /* generator */
    "0x2"
  },
  {
    "ietf-ike-grp-modp-3072",
    /* prime */
    "0x"
    "FFFFFFFFFFFFFFFFC90FDAA22168C234C4C6628B80DC1CD1"
    "29024E088A67CC74020BBEA63B139B22514A08798E3404DD"
    "EF9519B3CD3A431B302B0A6DF25F14374FE1356D6D51C245"
    "E485B576625E7EC6F44C42E9A637ED6B0BFF5CB6F406B7ED"
    "EE386BFB5A899FA5AE9F24117C4B1FE649286651ECE45B3D"
    "C2007CB8A163BF0598DA48361C55D39A69163FA8FD24CF5F"
    "83655D23DCA3AD961C62F356208552BB9ED529077096966D"
    "670C354E4ABC9804F1746C08CA18217C32905E462E36CE3B"
    "E39E772C180E86039B2783A2EC07A28FB5C55DF06F4C52C9"
    "DE2BCBF6955817183995497CEA956AE515D2261898FA0510"
    "15728E5A8AAAC42DAD33170D04507A33A85521ABDF1CBA64"
    "ECFB850458DBEF0A8AEA71575D060C7DB3970F85A6E1E4C7"
    "ABF5AE8CDB0933D71E8C94E04A25619DCEE3D2261AD2EE6B"
    "F12FFA06D98A0864D87602733EC86A64521F2B18177B200C"
    "BBE117577A615D6C770988C0BAD946E208E24FA074E5AB31"
    "43DB5BFCE0FD108E4B82D120A93AD2CAFFFFFFFFFFFFFFFF",
    /* LPF */
    NULL,
    /* generator */
    "0x2"
  },
  {
    "ietf-ike-grp-modp-4096",
    /* prime */
    "0x"
    "FFFFFFFFFFFFFFFFC90FDAA22168C234C4C6628B80DC1CD1"
    "29024E088A67CC74020BBEA63B139B22514A08798E3404DD"
    "EF9519B3CD3A431B302B0A6DF25F14374FE1356D6D51C245"
    "E485B576625E7EC6F44C42E9A637ED6B0BFF5CB6F406B7ED"
    "EE386BFB5A899FA5AE9F24117C4B1FE649286651ECE45B3D"
    "C2007CB8A163BF0598DA48361C55D39A69163FA8FD24CF5F"
    "83655D23DCA3AD961C62F356208552BB9ED529077096966D"
    "670C354E4ABC9804F1746C08CA18217C32905E462E36CE3B"
    "E39E772C180E86039B2783A2EC07A28FB5C55DF06F4C52C9"
    "DE2BCBF6955817183995497CEA956AE515D2261898FA0510"
    "15728E5A8AAAC42DAD33170D04507A33A85521ABDF1CBA64"
    "ECFB850458DBEF0A8AEA71575D060C7DB3970F85A6E1E4C7"
    "ABF5AE8CDB0933D71E8C94E04A25619DCEE3D2261AD2EE6B"
    "F12FFA06D98A0864D87602733EC86A64521F2B18177B200C"
    "BBE117577A615D6C770988C0BAD946E208E24FA074E5AB31"
    "43DB5BFCE0FD108E4B82D120A92108011A723C12A787E6D7"
    "88719A10BDBA5B2699C327186AF4E23C1A946834B6150BDA"
    "2583E9CA2AD44CE8DBBBC2DB04DE8EF92E8EFC141FBECAA6"
    "287C59474E6BC05D99B2964FA090C3A2233BA186515BE7ED"
    "1F612970CEE2D7AFB81BDD762170481CD0069127D5B05AA9"
    "93B4EA988D8FDDC186FFB7DC90A6C08F4DF435C934063199"
    "FFFFFFFFFFFFFFFF",
    /* LPF */
    NULL,
    /* generator */
    "0x2"
  },
  {
    "ietf-ike-grp-modp-6144",
    /* prime */
    "0x"
    "FFFFFFFFFFFFFFFFC90FDAA22168C234C4C6628B80DC1CD1"
    "29024E088A67CC74020BBEA63B139B22514A08798E3404DD"
    "EF9519B3CD3A431B302B0A6DF25F14374FE1356D6D51C245"
    "E485B576625E7EC6F44C42E9A637ED6B0BFF5CB6F406B7ED"
    "EE386BFB5A899FA5AE9F24117C4B1FE649286651ECE45B3D"
    "C2007CB8A163BF0598DA48361C55D39A69163FA8FD24CF5F"
    "83655D23DCA3AD961C62F356208552BB9ED529077096966D"
    "670C354E4ABC9804F1746C08CA18217C32905E462E36CE3B"
    "E39E772C180E86039B2783A2EC07A28FB5C55DF06F4C52C9"
    "DE2BCBF6955817183995497CEA956AE515D2261898FA0510"
    "15728E5A8AAAC42DAD33170D04507A33A85521ABDF1CBA64"
    "ECFB850458DBEF0A8AEA71575D060C7DB3970F85A6E1E4C7"
    "ABF5AE8CDB0933D71E8C94E04A25619DCEE3D2261AD2EE6B"
    "F12FFA06D98A0864D87602733EC86A64521F2B18177B200C"
    "BBE117577A615D6C770988C0BAD946E208E24FA074E5AB31"
    "43DB5BFCE0FD108E4B82D120A92108011A723C12A787E6D7"
    "88719A10BDBA5B2699C327186AF4E23C1A946834B6150BDA"
    "2583E9CA2AD44CE8DBBBC2DB04DE8EF92E8EFC141FBECAA6"
    "287C59474E6BC05D99B2964FA090C3A2233BA186515BE7ED"
    "1F612970CEE2D7AFB81BDD762170481CD0069127D5B05AA9"
    "93B4EA988D8FDDC186FFB7DC90A6C08F4DF435C934028492"
    "36C3FAB4D27C7026C1D4DCB2602646DEC9751E763DBA37BD"
    "F8FF9406AD9E530EE5DB382F413001AEB06A53ED9027D831"
    "179727B0865A8918DA3EDBEBCF9B14ED44CE6CBACED4BB1B"
    "DB7F1447E6CC254B332051512BD7AF426FB8F401378CD2BF"
    "5983CA01C64B92ECF032EA15D1721D03F482D7CE6E74FEF6"
    "D55E702F46980C82B5A84031900B1C9E59E7C97FBEC7E8F3"
    "23A97A7E36CC88BE0F1D45B7FF585AC54BD407B22B4154AA"
    "CC8F6D7EBF48E1D814CC5ED20F8037E0A79715EEF29BE328"
    "06A1D58BB7C5DA76F550AA3D8A1FBFF0EB19CCB1A313D55C"
    "DA56C9EC2EF29632387FE8D76E3C0468043E8F663F4860EE"
    "12BF2D5B0B7474D6E694F91E6DCC4024FFFFFFFFFFFFFFFF",
    /* LPF */
    NULL,
    /* generator */
    "0x2"
  },
  {
    "ietf-ike-grp-modp-8192",
    /* prime */
    "0x"
    "FFFFFFFFFFFFFFFFC90FDAA22168C234C4C6628B80DC1CD1"
    "29024E088A67CC74020BBEA63B139B22514A08798E3404DD"
    "EF9519B3CD3A431B302B0A6DF25F14374FE1356D6D51C245"
    "E485B576625E7EC6F44C42E9A637ED6B0BFF5CB6F406B7ED"
    "EE386BFB5A899FA5AE9F24117C4B1FE649286651ECE45B3D"
    "C2007CB8A163BF0598DA48361C55D39A69163FA8FD24CF5F"
    "83655D23DCA3AD961C62F356208552BB9ED529077096966D"
    "670C354E4ABC9804F1746C08CA18217C32905E462E36CE3B"
    "E39E772C180E86039B2783A2EC07A28FB5C55DF06F4C52C9"
    "DE2BCBF6955817183995497CEA956AE515D2261898FA0510"
    "15728E5A8AAAC42DAD33170D04507A33A85521ABDF1CBA64"
    "ECFB850458DBEF0A8AEA71575D060C7DB3970F85A6E1E4C7"
    "ABF5AE8CDB0933D71E8C94E04A25619DCEE3D2261AD2EE6B"
    "F12FFA06D98A0864D87602733EC86A64521F2B18177B200C"
    "BBE117577A615D6C770988C0BAD946E208E24FA074E5AB31"
    "43DB5BFCE0FD108E4B82D120A92108011A723C12A787E6D7"
    "88719A10BDBA5B2699C327186AF4E23C1A946834B6150BDA"
    "2583E9CA2AD44CE8DBBBC2DB04DE8EF92E8EFC141FBECAA6"
    "287C59474E6BC05D99B2964FA090C3A2233BA186515BE7ED"
    "1F612970CEE2D7AFB81BDD762170481CD0069127D5B05AA9"
    "93B4EA988D8FDDC186FFB7DC90A6C08F4DF435C934028492"
    "36C3FAB4D27C7026C1D4DCB2602646DEC9751E763DBA37BD"
    "F8FF9406AD9E530EE5DB382F413001AEB06A53ED9027D831"
    "179727B0865A8918DA3EDBEBCF9B14ED44CE6CBACED4BB1B"
    "DB7F1447E6CC254B332051512BD7AF426FB8F401378CD2BF"
    "5983CA01C64B92ECF032EA15D1721D03F482D7CE6E74FEF6"
    "D55E702F46980C82B5A84031900B1C9E59E7C97FBEC7E8F3"
    "23A97A7E36CC88BE0F1D45B7FF585AC54BD407B22B4154AA"
    "CC8F6D7EBF48E1D814CC5ED20F8037E0A79715EEF29BE328"
    "06A1D58BB7C5DA76F550AA3D8A1FBFF0EB19CCB1A313D55C"
    "DA56C9EC2EF29632387FE8D76E3C0468043E8F663F4860EE"
    "12BF2D5B0B7474D6E694F91E6DBE115974A3926F12FEE5E4"
    "38777CB6A932DF8CD8BEC4D073B931BA3BC832B68D9DD300"
    "741FA7BF8AFC47ED2576F6936BA424663AAB639C5AE4F568"
    "3423B4742BF1C978238F16CBE39D652DE3FDB8BEFC848AD9"
    "22222E04A4037C0713EB57A81A23F0C73473FC646CEA306B"
    "4BCBC8862F8385DDFA9D4B7FA2C087E879683303ED5BDD3A"
    "062B3CF5B3A278A66D2A13F83F44F82DDF310EE074AB6A36"
    "4597E899A0255DC164F31CC50846851DF9AB48195DED7EA1"
    "B1D510BD7EE74D73FAF36BC31ECFA268359046F4EB879F92"
    "4009438B481C6CD7889A002ED5EE382BC9190DA6FC026E47"
    "9558E4475677E9AA9E3050E2765694DFC81F56E880B96E71"
    "60C980DD98EDD3DFFFFFFFFFFFFFFFFF",
    /* LPF */
    NULL,
    /* generator */
    "0x2"
  },
  {
    "ietf-ike-grp-modp-12288",
    /* prime */
    "0x"
    "FFFFFFFFFFFFFFFFC90FDAA22168C234C4C6628B80DC1CD129024E088A67CC74020B"
    "BEA63B139B22514A08798E3404DDEF9519B3CD3A431B302B0A6DF25F14374FE1356D"
    "6D51C245E485B576625E7EC6F44C42E9A637ED6B0BFF5CB6F406B7EDEE386BFB5A89"
    "9FA5AE9F24117C4B1FE649286651ECE45B3DC2007CB8A163BF0598DA48361C55D39A"
    "69163FA8FD24CF5F83655D23DCA3AD961C62F356208552BB9ED529077096966D670C"
    "354E4ABC9804F1746C08CA18217C32905E462E36CE3BE39E772C180E86039B2783A2"
    "EC07A28FB5C55DF06F4C52C9DE2BCBF6955817183995497CEA956AE515D2261898FA"
    "051015728E5A8AAAC42DAD33170D04507A33A85521ABDF1CBA64ECFB850458DBEF0A"
    "8AEA71575D060C7DB3970F85A6E1E4C7ABF5AE8CDB0933D71E8C94E04A25619DCEE3"
    "D2261AD2EE6BF12FFA06D98A0864D87602733EC86A64521F2B18177B200CBBE11757"
    "7A615D6C770988C0BAD946E208E24FA074E5AB3143DB5BFCE0FD108E4B82D120A921"
    "08011A723C12A787E6D788719A10BDBA5B2699C327186AF4E23C1A946834B6150BDA"
    "2583E9CA2AD44CE8DBBBC2DB04DE8EF92E8EFC141FBECAA6287C59474E6BC05D99B2"
    "964FA090C3A2233BA186515BE7ED1F612970CEE2D7AFB81BDD762170481CD0069127"
    "D5B05AA993B4EA988D8FDDC186FFB7DC90A6C08F4DF435C93402849236C3FAB4D27C"
    "7026C1D4DCB2602646DEC9751E763DBA37BDF8FF9406AD9E530EE5DB382F413001AE"
    "B06A53ED9027D831179727B0865A8918DA3EDBEBCF9B14ED44CE6CBACED4BB1BDB7F"
    "1447E6CC254B332051512BD7AF426FB8F401378CD2BF5983CA01C64B92ECF032EA15"
    "D1721D03F482D7CE6E74FEF6D55E702F46980C82B5A84031900B1C9E59E7C97FBEC7"
    "E8F323A97A7E36CC88BE0F1D45B7FF585AC54BD407B22B4154AACC8F6D7EBF48E1D8"
    "14CC5ED20F8037E0A79715EEF29BE32806A1D58BB7C5DA76F550AA3D8A1FBFF0EB19"
    "CCB1A313D55CDA56C9EC2EF29632387FE8D76E3C0468043E8F663F4860EE12BF2D5B"
    "0B7474D6E694F91E6DBE115974A3926F12FEE5E438777CB6A932DF8CD8BEC4D073B9"
    "31BA3BC832B68D9DD300741FA7BF8AFC47ED2576F6936BA424663AAB639C5AE4F568"
    "3423B4742BF1C978238F16CBE39D652DE3FDB8BEFC848AD922222E04A4037C0713EB"
    "57A81A23F0C73473FC646CEA306B4BCBC8862F8385DDFA9D4B7FA2C087E879683303"
    "ED5BDD3A062B3CF5B3A278A66D2A13F83F44F82DDF310EE074AB6A364597E899A025"
    "5DC164F31CC50846851DF9AB48195DED7EA1B1D510BD7EE74D73FAF36BC31ECFA268"
    "359046F4EB879F924009438B481C6CD7889A002ED5EE382BC9190DA6FC026E479558"
    "E4475677E9AA9E3050E2765694DFC81F56E880B96E7160C980DD98A573EA4472065A"
    "139CD2906CD1CB729EC52A5286D44014A694CA457583D5CFEF26F1B90AD8291DA079"
    "9D00022E9BED55C6FA47FCA5BB1ACA8376456D98D94879EE7E6DBFCD014BB1615599"
    "14EC0B576A67E3E8422E91E65BA141DA92DE9C3A6D6CCA5136DD424BB1064988EB5B"
    "A9AC1269F7DF673B982E23FB6C99BB2AA31C5A6685FFD599149B30AC67B8464D80A9"
    "5D42530A681644D039060E8F8FD5262696D0A7595AE3F935A67DCFF5A874A701FBFA"
    "0C3D534B4E39BC09577053374821A11C3AC998E0BA718087B317825A1ACFCFAEBBF2"
    "4F25C6051ADA9C285A1FCD6114A838A1ADE714C16A9401CDCF81E1071FF7AB97239F"
    "513B15C5BCAE2C0EB68DFC1403037C0707C100802CFFEB833D468F2D5D2C8960DE96"
    "3702486F746444FE5F2A4BFDA50C91DCC8BD51C04EB979604DF0B6B7322D5D8D26BC"
    "F769EA51185183F400C3BB3231CFA91D4790788E33664EFA838BCCA02EE8460FACCC"
    "539522CE13DB6E421BD08340FD82812FCB2E04A40925DF1E559E6C1CAF2BE26BF7A6"
    "9DC7F664C2042CE2EB84B733CFCB95449C87CB9ADC491406B779A7E13361DE9611C6"
    "1D023685EF27E6AF3A52DF633B1EBB0EB6E1477E98C250D9B11930F4BBC70611CC85"
    "76423750CECDC930AE8584A85350CA9971145425000084CEB9375C77FE27840C5395"
    "606B1DF597C44666C10D55BC75E8F1DACF04460ED64929427CA3F9BB666DCDE2FFFF"
    "FFFFFFFFFFFF",
    /* LPF */
    NULL,
    /* generator */
    "0x2"
  },
  {
    "ietf-ike-grp-modp-16384",
    /* prime */
    "0x"
    "FFFFFFFFFFFFFFFFC90FDAA22168C234C4C6628B80DC1CD129024E088A67CC74020B"
    "BEA63B139B22514A08798E3404DDEF9519B3CD3A431B302B0A6DF25F14374FE1356D"
    "6D51C245E485B576625E7EC6F44C42E9A637ED6B0BFF5CB6F406B7EDEE386BFB5A89"
    "9FA5AE9F24117C4B1FE649286651ECE45B3DC2007CB8A163BF0598DA48361C55D39A"
    "69163FA8FD24CF5F83655D23DCA3AD961C62F356208552BB9ED529077096966D670C"
    "354E4ABC9804F1746C08CA18217C32905E462E36CE3BE39E772C180E86039B2783A2"
    "EC07A28FB5C55DF06F4C52C9DE2BCBF6955817183995497CEA956AE515D2261898FA"
    "051015728E5A8AAAC42DAD33170D04507A33A85521ABDF1CBA64ECFB850458DBEF0A"
    "8AEA71575D060C7DB3970F85A6E1E4C7ABF5AE8CDB0933D71E8C94E04A25619DCEE3"
    "D2261AD2EE6BF12FFA06D98A0864D87602733EC86A64521F2B18177B200CBBE11757"
    "7A615D6C770988C0BAD946E208E24FA074E5AB3143DB5BFCE0FD108E4B82D120A921"
    "08011A723C12A787E6D788719A10BDBA5B2699C327186AF4E23C1A946834B6150BDA"
    "2583E9CA2AD44CE8DBBBC2DB04DE8EF92E8EFC141FBECAA6287C59474E6BC05D99B2"
    "964FA090C3A2233BA186515BE7ED1F612970CEE2D7AFB81BDD762170481CD0069127"
    "D5B05AA993B4EA988D8FDDC186FFB7DC90A6C08F4DF435C93402849236C3FAB4D27C"
    "7026C1D4DCB2602646DEC9751E763DBA37BDF8FF9406AD9E530EE5DB382F413001AE"
    "B06A53ED9027D831179727B0865A8918DA3EDBEBCF9B14ED44CE6CBACED4BB1BDB7F"
    "1447E6CC254B332051512BD7AF426FB8F401378CD2BF5983CA01C64B92ECF032EA15"
    "D1721D03F482D7CE6E74FEF6D55E702F46980C82B5A84031900B1C9E59E7C97FBEC7"
    "E8F323A97A7E36CC88BE0F1D45B7FF585AC54BD407B22B4154AACC8F6D7EBF48E1D8"
    "14CC5ED20F8037E0A79715EEF29BE32806A1D58BB7C5DA76F550AA3D8A1FBFF0EB19"
    "CCB1A313D55CDA56C9EC2EF29632387FE8D76E3C0468043E8F663F4860EE12BF2D5B"
    "0B7474D6E694F91E6DBE115974A3926F12FEE5E438777CB6A932DF8CD8BEC4D073B9"
    "31BA3BC832B68D9DD300741FA7BF8AFC47ED2576F6936BA424663AAB639C5AE4F568"
    "3423B4742BF1C978238F16CBE39D652DE3FDB8BEFC848AD922222E04A4037C0713EB"
    "57A81A23F0C73473FC646CEA306B4BCBC8862F8385DDFA9D4B7FA2C087E879683303"
    "ED5BDD3A062B3CF5B3A278A66D2A13F83F44F82DDF310EE074AB6A364597E899A025"
    "5DC164F31CC50846851DF9AB48195DED7EA1B1D510BD7EE74D73FAF36BC31ECFA268"
    "359046F4EB879F924009438B481C6CD7889A002ED5EE382BC9190DA6FC026E479558"
    "E4475677E9AA9E3050E2765694DFC81F56E880B96E7160C980DD98A573EA4472065A"
    "139CD2906CD1CB729EC52A5286D44014A694CA457583D5CFEF26F1B90AD8291DA079"
    "9D00022E9BED55C6FA47FCA5BB1ACA8376456D98D94879EE7E6DBFCD014BB1615599"
    "14EC0B576A67E3E8422E91E65BA141DA92DE9C3A6D6CCA5136DD424BB1064988EB5B"
    "A9AC1269F7DF673B982E23FB6C99BB2AA31C5A6685FFD599149B30AC67B8464D80A9"
    "5D42530A681644D039060E8F8FD5262696D0A7595AE3F935A67DCFF5A874A701FBFA"
    "0C3D534B4E39BC09577053374821A11C3AC998E0BA718087B317825A1ACFCFAEBBF2"
    "4F25C6051ADA9C285A1FCD6114A838A1ADE714C16A9401CDCF81E1071FF7AB97239F"
    "513B15C5BCAE2C0EB68DFC1403037C0707C100802CFFEB833D468F2D5D2C8960DE96"
    "3702486F746444FE5F2A4BFDA50C91DCC8BD51C04EB979604DF0B6B7322D5D8D26BC"
    "F769EA51185183F400C3BB3231CFA91D4790788E33664EFA838BCCA02EE8460FACCC"
    "539522CE13DB6E421BD08340FD82812FCB2E04A40925DF1E559E6C1CAF2BE26BF7A6"
    "9DC7F664C2042CE2EB84B733CFCB95449C87CB9ADC491406B779A7E13361DE9611C6"
    "1D023685EF27E6AF3A52DF633B1EBB0EB6E1477E98C250D9B11930F4BBC70611CC85"
    "76423750CECDC930AE8584A85350CA9971145425000084CEB9375C77FE27840C5395"
    "606B1DF597C44666C10D55BC75E8F1DACF04460ED64929427CA3F9BB65FC7EFEA7AE"
    "AFCB07854F1BA1B8D15C3ABA5BEC61839782968F8AACDDC7F9C7138F41BE8A59772E"
    "6679C743E00FA2759499B2094B93325E27042CDAB18543AEA538BA9E297F0F14C782"
    "8B7D3CBDD3A9CD874ACF464E4983C6709E581488E9C23DC4C4ADBAEB7F9BBAB0C7D9"
    "B8EF1165699EF220EC5FCDF440633FCA30CCB77BEF9B16A9595608615A2AE600BBB3"
    "A943F6CBE54ECABBDF6B56DB8BE105486D8A0A41D85C3B3751DD5867C54404F32A0C"
    "3AD86F6580CD3F87AA80D8F3ED5CD724131C288E7567A782F2EAB7853BB321AF1818"
    "8B29E72AD72AECBCE11B9922C7ABC66F7C32A808DA6E5956AED4101A168C8F0AAD2C"
    "CC67BA7570086E3DE6D502C661D7E826657DE65F988F5F6A3E0DE226A5F8CB5DC47B"
    "64D7C59A04A0438D620A71F987F5A5B7B7E85E162EA655FD612946C89C98E6E0F0FF"
    "C6B091A5B36CC2BAD4CB8C1523F652391B6F0C4A163AFCBBCD31BFFABF8A3B587B9F"
    "0F1CD75285367A192DF8D0841745080F84F8117BB8ADA8EAAAFAB6DB13C57EB2D3F4"
    "31D0BD10BBDAAEED5953CEC75073484176079E67A1A15371F912D1DA8F60589433D8"
    "A87C96E34991BF2220E83071EDA8DFC54930DA72DD2491E12282D5A4ACA14256EFC0"
    "2B4652274518AC5D08E083801610A34A83157D7A876B7D0F88CFDC184CDCBC24A364"
    "DF907597FB3C5B088EF6DF378DD672FB9D1810217CA9F39DCC9BA981E0210985723A"
    "FFFFFFFFFFFFFFFF",
    /* LPF */
    NULL,
    /* generator */
    "0x2"
  },
#endif /* SSHDIST_CRYPT_DL_IKE_GROUPS */

  /* RFC5114 groups. */
  {
    /* 1024-bit MODP Group with 160-bit Prime Order Subgroup */
    "ietf-rfc5114-2-1-modp-1024-160",
    /* p */
    "0x"
    "B10B8F96 A080E01D DE92DE5E AE5D54EC 52C99FBC FB06A3C6"
    "9A6A9DCA 52D23B61 6073E286 75A23D18 9838EF1E 2EE652C0"
    "13ECB4AE A9061123 24975C3C D49B83BF ACCBDD7D 90C4BD70"
    "98488E9C 219A7372 4EFFD6FA E5644738 FAA31A4F F55BCCC0"
    "A151AF5F 0DC8B4BD 45BF37DF 365C1A65 E68CFDA7 6D4DA708"
    "DF1FB2BC 2E4A4371",
    /* q */
    "0x"
    "F518AA87 81A8DF27 8ABA4E7D 64B7CB9D 49462353",
    /* g */
    "0x"
    "A4D1CBD5 C3FD3412 6765A442 EFB99905 F8104DD2 58AC507F"
    "D6406CFF 14266D31 266FEA1E 5C41564B 777E690F 5504F213"
    "160217B4 B01B886A 5E91547F 9E2749F4 D7FBD7D3 B9A92EE1"
    "909D0D22 63F80A76 A6A24C08 7A091F53 1DBF0A01 69B6A28A"
    "D662A4D1 8E73AFA3 2D779D59 18D08BC8 858F4DCE F97C2A24"
    "855E6EEB 22B3B2E5"
  },
  {
    /* 2048-bit MODP Group with 224-bit Prime Order Subgroup */
    "ietf-rfc5114-2-2-modp-2048-224",
    /* p */
    "0x"
    "AD107E1E 9123A9D0 D660FAA7 9559C51F A20D64E5 683B9FD1"
    "B54B1597 B61D0A75 E6FA141D F95A56DB AF9A3C40 7BA1DF15"
    "EB3D688A 309C180E 1DE6B85A 1274A0A6 6D3F8152 AD6AC212"
    "9037C9ED EFDA4DF8 D91E8FEF 55B7394B 7AD5B7D0 B6C12207"
    "C9F98D11 ED34DBF6 C6BA0B2C 8BBC27BE 6A00E0A0 B9C49708"
    "B3BF8A31 70918836 81286130 BC8985DB 1602E714 415D9330"
    "278273C7 DE31EFDC 7310F712 1FD5A074 15987D9A DC0A486D"
    "CDF93ACC 44328387 315D75E1 98C641A4 80CD86A1 B9E587E8"
    "BE60E69C C928B2B9 C52172E4 13042E9B 23F10B0E 16E79763"
    "C9B53DCF 4BA80A29 E3FB73C1 6B8E75B9 7EF363E2 FFA31F71"
    "CF9DE538 4E71B81C 0AC4DFFE 0C10E64F",
    /* q */
    "0x"
    "801C0D34 C58D93FE 99717710 1F80535A 4738CEBC BF389A99"
    "B36371EB",
    /* g */
    "0x"
    "AC4032EF 4F2D9AE3 9DF30B5C 8FFDAC50 6CDEBE7B 89998CAF"
    "74866A08 CFE4FFE3 A6824A4E 10B9A6F0 DD921F01 A70C4AFA"
    "AB739D77 00C29F52 C57DB17C 620A8652 BE5E9001 A8D66AD7"
    "C1766910 1999024A F4D02727 5AC1348B B8A762D0 521BC98A"
    "E2471504 22EA1ED4 09939D54 DA7460CD B5F6C6B2 50717CBE"
    "F180EB34 118E98D1 19529A45 D6F83456 6E3025E3 16A330EF"
    "BB77A86F 0C1AB15B 051AE3D4 28C8F8AC B70A8137 150B8EEB"
    "10E183ED D19963DD D9E263E4 770589EF 6AA21E7F 5F2FF381"
    "B539CCE3 409D13CD 566AFBB4 8D6C0191 81E1BCFE 94B30269"
    "EDFE72FE 9B6AA4BD 7B5A0F1C 71CFFF4C 19C418E1 F6EC0179"
    "81BC087F 2A7065B3 84B890D3 191F2BFA"
  },
  
  {
    /* 2048-bit MODP Group with 256-bit Prime Order Subgroup */
    "ietf-rfc5114-2-3-modp-2048-256",
    /* p */
    "0x"
    "87A8E61D B4B6663C FFBBD19C 65195999 8CEEF608 660DD0F2"
    "5D2CEED4 435E3B00 E00DF8F1 D61957D4 FAF7DF45 61B2AA30"
    "16C3D911 34096FAA 3BF4296D 830E9A7C 209E0C64 97517ABD"
    "5A8A9D30 6BCF67ED 91F9E672 5B4758C0 22E0B1EF 4275BF7B"
    "6C5BFC11 D45F9088 B941F54E B1E59BB8 BC39A0BF 12307F5C"
    "4FDB70C5 81B23F76 B63ACAE1 CAA6B790 2D525267 35488A0E"
    "F13C6D9A 51BFA4AB 3AD83477 96524D8E F6A167B5 A41825D9"
    "67E144E5 14056425 1CCACB83 E6B486F6 B3CA3F79 71506026"
    "C0B857F6 89962856 DED4010A BD0BE621 C3A3960A 54E710C3"
    "75F26375 D7014103 A4B54330 C198AF12 6116D227 6E11715F"
    "693877FA D7EF09CA DB094AE9 1E1A1597",
    /* q */
    "0x"
    "8CF83642 A709A097 B4479976 40129DA2 99B1A47D 1EB3750B"
    "A308B0FE 64F5FBD3",
    /* g */
    "0x"
    "3FB32C9B 73134D0B 2E775066 60EDBD48 4CA7B18F 21EF2054"
    "07F4793A 1A0BA125 10DBC150 77BE463F FF4FED4A AC0BB555"
    "BE3A6C1B 0C6B47B1 BC3773BF 7E8C6F62 901228F8 C28CBB18"
    "A55AE313 41000A65 0196F931 C77A57F2 DDF463E5 E9EC144B"
    "777DE62A AAB8A862 8AC376D2 82D6ED38 64E67982 428EBC83"
    "1D14348F 6F2F9193 B5045AF2 767164E1 DFC967C1 FB3F2E55"
    "A4BD1BFF E83B9C80 D052B985 D182EA0A DB2A3B73 13D3FE14"
    "C8484B1E 052588B9 B7D2BBD2 DF016199 ECD06E15 57CD0915"
    "B3353BBB 64E0EC37 7FD02837 0DF92B52 C7891428 CDC67EB6"
    "184B523D 1DB246C3 2F630784 90F00EF8 D647D148 D4795451"
    "5E2327CF EF98C582 664B4C0F 6CC41659"
  },

  { NULL }
};

char *ssh_dlp_param_get_predefined_groups(void)
{
  char *list = NULL;
  SshBufferStruct buffer;
  unsigned int i;

  ssh_buffer_init(&buffer);
  for (i = 0; ssh_dlp_fixed_params[i].name; i++)
    {
      if (ssh_buffer_len(&buffer) > 0)
        {
          if (ssh_buffer_append(&buffer, (unsigned char *) ",", 1)
              != SSH_BUFFER_OK)
            goto failure;
        }
      if (ssh_buffer_append(&buffer,
                            (unsigned char *) ssh_dlp_fixed_params[i].name,
                            strlen(ssh_dlp_fixed_params[i].name))
          != SSH_BUFFER_OK)
        goto failure;
    }

  if (ssh_buffer_append(&buffer, (unsigned char *) "\0", 1) == SSH_BUFFER_OK)
    list = ssh_strdup(ssh_buffer_ptr(&buffer));

 failure:
  ssh_buffer_uninit(&buffer);
  return list;
}

Boolean ssh_dlp_set_param(const char *name, const char **outname,
                          SshMPInteger p, SshMPInteger q, SshMPInteger g)
{
  int i;

  if (name == NULL)
    name = SSH_DEFAULT_DL_MODP_PARAM_NAME;

  for (i = 0; ssh_dlp_fixed_params[i].name; i++)
    {
      if (strcmp(ssh_dlp_fixed_params[i].name, name) == 0)
        break;
    }
  if (ssh_dlp_fixed_params[i].name == NULL)
    return FALSE;

  ssh_mprz_set_str(p, ssh_dlp_fixed_params[i].p, 0);
  ssh_mprz_set_str(g, ssh_dlp_fixed_params[i].g, 0);

  /* If the q string is NULL, set q to (p-1)/2 */
  if (ssh_dlp_fixed_params[i].q == NULL)
    {
      ssh_mprz_set(q, p);
      ssh_mprz_sub_ui(q, q, 1);
      ssh_mprz_div_ui(q, q, 2);
    }
  else
    {
      ssh_mprz_set_str(q, ssh_dlp_fixed_params[i].q, 0);
    }

  if (ssh_mprz_isnan(p) || ssh_mprz_isnan(q) || ssh_mprz_isnan(g))
    {
      ssh_mprz_clear(p);
      ssh_mprz_clear(q);
      ssh_mprz_clear(g);
      return FALSE;
    }

  *outname = ssh_dlp_fixed_params[i].name;
  return TRUE;
}

Boolean ssh_dlp_is_predefined_group(SshMPInteger p, SshMPInteger q,
                                    SshMPInteger g)
{
  unsigned int i;
  SshMPIntegerStruct P, Q, G;

  ssh_mprz_init(&P);
  ssh_mprz_init(&Q);
  ssh_mprz_init(&G);

  for (i = 0; ssh_dlp_fixed_params[i].name; i++)
    {
      ssh_mprz_set_str(&P, ssh_dlp_fixed_params[i].p, 0);
      ssh_mprz_set_str(&G, ssh_dlp_fixed_params[i].g, 0);

      /* If the q string is NULL, set q to (p-1)/2 */
      if (ssh_dlp_fixed_params[i].q == NULL)
        {
          ssh_mprz_set(&Q, &P);
          ssh_mprz_sub_ui(&Q, &Q, 1);
          ssh_mprz_div_ui(&Q, &Q, 2);
        }
      else
        {
          ssh_mprz_set_str(&Q, ssh_dlp_fixed_params[i].q, 0);
        }

      if ((ssh_mprz_cmp(p, &P) == 0) &&
          (ssh_mprz_cmp(q, &Q) == 0) &&
          (ssh_mprz_cmp(g, &G) == 0))
        break;
    }

  ssh_mprz_clear(&P);
  ssh_mprz_clear(&Q);
  ssh_mprz_clear(&G);

  if (ssh_dlp_fixed_params[i].name == NULL)
    return FALSE;

  return TRUE;
}

/* dlfix.c */
#endif /* SSHDIST_CRYPT_DL */
