/*
  sshstr.c

  Author Mika Kojo <mkojo@ssh.fi>
  Copyright:
          Copyright (c) 2002, 2003 SFNT Finland Oy.
                All rights reserved.

  Implementation of simple character set and string handling routines. These
  are needed in the X.509 PKIX compatible library.

  We take the following approach with the character sets:

    1. We work based on the 16-bit BMP set which is a subset of
       UCS-4 (it is called also UCS-2).

    2. All 8-bit sets are defined by referring to BMP (UCS-2).

    3. Only 8-bit (or 7-bit) tables are defined for handling of
       special charsets. These point to the 16-bit set.

    4. Mappings between 8-bit sets are done by construction suitable
       16-bit tables on the fly and doing the transformation there.
       If possible, that is.

    5. Canonical transformation is as follows:

       a) Remove all whitespace that doesn't separate characters.
       b) Make all letters lowercase (if possible, this is defined to
          work only for ISO 646 or related).
*/

#include "sshincludes.h"
#include "sshstr.h"

#define SSH_DEBUG_MODULE "SshStr"































/* The string data structure. */
struct SshStrRec
{
  SshCharset      charset;      /* character set in use. */
  unsigned  int   bits;         /* string bits. */
  size_t          str_length;
  union {
    unsigned char  *str_8bit;
    SshUInt16      *str_16bit;
    SshUInt32      *str_32bit;
  } ob;
  size_t          mem_length;   /* total number of bytes allocated. */
};

/* Basic string interface. */

/* Note: It would be possible to write the UTF-8 encoding and decoding
   with using string routine append capability rather than having
   adhoc buffer routines as above I do. */

static Boolean
ssh_str_realloc(SshStr str, size_t new_length)
{
  SshUInt32 *newp = NULL;

#define STR_REALLOC 256

  if (str->mem_length < new_length)
    {
      if ((newp =
           ssh_malloc((new_length + STR_REALLOC) * (str->bits / 8)))
          != NULL)
        {
          str->mem_length = new_length + STR_REALLOC;
          switch (str->bits)
            {
            case 8:
              memcpy(newp,
                     str->ob.str_8bit,
                     str->str_length * sizeof(unsigned char));
              ssh_free(str->ob.str_8bit);
              str->ob.str_8bit = (unsigned char *)newp;
              break;

            case 16:
              memcpy(newp,
                     str->ob.str_16bit,
                     str->str_length * sizeof(SshUInt16));
              ssh_free(str->ob.str_16bit);
              str->ob.str_16bit = (SshUInt16 *)newp;
              break;

            case 32:
              memcpy(newp,
                     str->ob.str_32bit,
                     str->str_length * sizeof(SshUInt32));
              ssh_free(str->ob.str_32bit);
              str->ob.str_32bit = (SshUInt32 *)newp;
              break;

            default:
              SSH_DEBUG(SSH_D_ERROR,
                        ("Failed to convert charset %u (%u bit chars).",
                         str->charset, str->bits));
              break;
            }
        }
      if (newp == NULL)
        return FALSE;
    }
  return TRUE;
}

Boolean ssh_str_get_letter(SshStr str, size_t pos, SshUInt32 *letter)
{
  if (str == NULL || pos >= str->str_length)
    return FALSE;

  switch (str->bits)
    {
    case 8:
      *letter = str->ob.str_8bit[pos];
      break;
    case 16:
      *letter = str->ob.str_16bit[pos];
      break;
    case 32:
      *letter = str->ob.str_32bit[pos];
      break;
    default:
      return FALSE;
    }
  return TRUE;
}

/* Function to add a letter to the string. */
static Boolean ssh_str_append_letter(SshStr str, SshUInt32 letter)
{
  SSH_PRECOND(str != NULL);

  /* Make sure there is always space for extra zero at the end of the
     string. */
  if (ssh_str_realloc(str, str->str_length + 2))
    {
      switch (str->bits)
        {
        case 8:
          if ((letter & 0xff00) != 0)
            {
              str->ob.str_8bit[str->str_length] = (letter >> 8) & 0xff;
              str->str_length++;
            }
          str->ob.str_8bit[str->str_length] = letter & 0xff;
          break;
        case 16:
          str->ob.str_16bit[str->str_length] = letter & 0xffff;
          break;
        case 32:
          str->ob.str_32bit[str->str_length] = letter;
          break;
        default:
          SSH_DEBUG(SSH_D_ERROR,
                    ("Character set %u (%u bit chars) unsupported.",
                     str->charset, str->bits));
          return FALSE;
        }
      str->str_length++;
      return TRUE;
    }
  return FALSE;
}

SshStr ssh_str_allocate(SshCharset charset,
                        size_t initial_memory)
{
  SshStr new_str = ssh_calloc(1, sizeof(*new_str));

  if (!new_str)
    return NULL;

  new_str->charset    = charset;
  new_str->bits       = 0;
  new_str->str_length = 0;
  new_str->mem_length = 0;

  switch (charset)
    {
    case SSH_CHARSET_PRINTABLE:
    case SSH_CHARSET_VISIBLE:
    case SSH_CHARSET_US_ASCII:
    case SSH_CHARSET_ISO_8859_1:
    case SSH_CHARSET_ISO_8859_2:
    case SSH_CHARSET_ISO_8859_3:
    case SSH_CHARSET_ISO_8859_4:
    case SSH_CHARSET_ISO_8859_15:
    case SSH_CHARSET_T61:
      /* Bit size. */
      new_str->bits = 8;

      if (initial_memory == 0)
        break;

      new_str->mem_length    = initial_memory;
      if ((new_str->ob.str_8bit = ssh_malloc(initial_memory)) == NULL)
        {
          ssh_free(new_str);
          return NULL;
        }
      break;
    case SSH_CHARSET_BMP:
      /* Bit size. */
      new_str->bits = 16;

      if (initial_memory == 0)
        break;

      new_str->mem_length    = initial_memory;
      if ((new_str->ob.str_16bit  =
           ssh_malloc(initial_memory * sizeof(SshUInt16))) == NULL)
        {
          ssh_free(new_str);
          return NULL;
        }
      break;
    case SSH_CHARSET_UNIVERSAL:
    case SSH_CHARSET_UTF8:
      /* Bit size. */
      new_str->bits = 32;

      if (initial_memory == 0)
        break;

      new_str->mem_length    = initial_memory;
      if ((new_str->ob.str_32bit  =
           ssh_malloc(initial_memory * sizeof(SshUInt32))) == NULL)
        {
          ssh_free(new_str);
          return NULL;
        }
      break;
    default:
      ssh_fatal("ssh_str_allocate: unknown character set %u.", charset);
      break;
    }
  return new_str;
}

unsigned char *ssh_str_get_data(SshStr in_str, size_t *out_str_length)
{
  unsigned char *out_str = NULL;

  if (in_str == NULL)
    return NULL;

  switch (in_str->bits)
    {
    case 8:
      out_str         = in_str->ob.str_8bit;
      *out_str_length = in_str->str_length;
      break;

    default:
      ssh_fatal("ssh_str_get_data: "
                "cannot convert directly %u bit letters (charset %u) to "
                "'unsigned char' type.", in_str->bits, in_str->charset);
      break;
    }
  return out_str;
}

/* Free the actual data. */
void ssh_str_free(SshStr str)
{
  if (str == NULL)
    return;
  switch (str->bits)
    {
    case 8:
      ssh_free(str->ob.str_8bit);
      break;
    case 16:
      ssh_free(str->ob.str_16bit);
      break;
    case 32:
      ssh_free(str->ob.str_32bit);
      break;
    default:
      ssh_fatal("unknown %d bit char set.", str->bits);
      break;
    }
  ssh_free(str);
}

/* This function frees the wrapper data structure only, not the
   underlying string. */
void ssh_str_free_wrapper(SshStr str)
{
  if (str == NULL)
    return;
  ssh_free(str);
}

/* Tables:

   These tables are given to convert some particular set of characters
   from 8-bit to 16-bit BMP set. This mapping might not always be
   exactly correct. However, it is assumed that not all features of
   some exotic character sets are used by the application.

   0xffff denotes unsupported letter.

   Some of the space could be saved with more careful analysis of the
   contents of the tables. That might even speed up the searching a
   bit. */

typedef struct
{
  SshUInt16 unicode;
  SshUInt16 mapped;
} SshCharsetMap;

/* Numerical table. */

static const SshUInt16 ssh_charset_numerical_string[] =
  {
    0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
    0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
    0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
    0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
    0x0020,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
    0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
    0x0030,0x0031,0x0032,0x0033,0x0034,0x0035,0x0036,0x0037,
    0x0038,0x0039,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff
  };

/* Printable string table. */
static const SshUInt16 ssh_charset_printable_string[] =
  {
    0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
    0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
    0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
    0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
    0x0020,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0x0027,
    0x0028,0x0029,0xffff,0x002b,0x002c,0x002d,0x002e,0x002f,
    0x0030,0x0031,0x0032,0x0033,0x0034,0x0035,0x0036,0x0037,
    0x0038,0x0039,0x003a,0xffff,0xffff,0x003d,0xffff,0x003f,
    0xffff,0x0041,0x0042,0x0043,0x0044,0x0045,0x0046,0x0047,
    0x0048,0x0049,0x004a,0x004b,0x004c,0x004d,0x004e,0x004f,
    0x0050,0x0051,0x0052,0x0053,0x0054,0x0055,0x0056,0x0057,
    0x0058,0x0059,0x005a,0xffff,0xffff,0xffff,0xffff,0xffff,
    0xffff,0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,
    0x0068,0x0069,0x006a,0x006b,0x006c,0x006d,0x006e,0x006f,
    0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,
    0x0078,0x0079,0x007a,0xffff,0xffff,0xffff,0xffff,0xffff
  };

static const SshCharsetMap ssh_charset_printable_string_map[] =
  {
    { 0x0020,0x0020 },{ 0x0027,0x0027 },{ 0x0028,0x0028 },{ 0x0029,0x0029 },
    { 0x002b,0x002b },{ 0x002c,0x002c },{ 0x002d,0x002d },{ 0x002e,0x002e },
    { 0x002f,0x002f },{ 0x0030,0x0030 },{ 0x0031,0x0031 },{ 0x0032,0x0032 },
    { 0x0033,0x0033 },{ 0x0034,0x0034 },{ 0x0035,0x0035 },{ 0x0036,0x0036 },
    { 0x0037,0x0037 },{ 0x0038,0x0038 },{ 0x0039,0x0039 },{ 0x003a,0x003a },
    { 0x003d,0x003d },{ 0x003f,0x003f },{ 0x0041,0x0041 },{ 0x0042,0x0042 },
    { 0x0043,0x0043 },{ 0x0044,0x0044 },{ 0x0045,0x0045 },{ 0x0046,0x0046 },
    { 0x0047,0x0047 },{ 0x0048,0x0048 },{ 0x0049,0x0049 },{ 0x004a,0x004a },
    { 0x004b,0x004b },{ 0x004c,0x004c },{ 0x004d,0x004d },{ 0x004e,0x004e },
    { 0x004f,0x004f },{ 0x0050,0x0050 },{ 0x0051,0x0051 },{ 0x0052,0x0052 },
    { 0x0053,0x0053 },{ 0x0054,0x0054 },{ 0x0055,0x0055 },{ 0x0056,0x0056 },
    { 0x0057,0x0057 },{ 0x0058,0x0058 },{ 0x0059,0x0059 },{ 0x005a,0x005a },
    { 0x0061,0x0061 },{ 0x0062,0x0062 },{ 0x0063,0x0063 },{ 0x0064,0x0064 },
    { 0x0065,0x0065 },{ 0x0066,0x0066 },{ 0x0067,0x0067 },{ 0x0068,0x0068 },
    { 0x0069,0x0069 },{ 0x006a,0x006a },{ 0x006b,0x006b },{ 0x006c,0x006c },
    { 0x006d,0x006d },{ 0x006e,0x006e },{ 0x006f,0x006f },{ 0x0070,0x0070 },
    { 0x0071,0x0071 },{ 0x0072,0x0072 },{ 0x0073,0x0073 },{ 0x0074,0x0074 },
    { 0x0075,0x0075 },{ 0x0076,0x0076 },{ 0x0077,0x0077 },{ 0x0078,0x0078 },
    { 0x0079,0x0079 },{ 0x007a,0x007a },{ 0xffff,0x001f },{ 0xffff,0x0000 },
    { 0xffff,0x0021 },{ 0xffff,0x0022 },{ 0xffff,0x0023 },{ 0xffff,0x0024 },
    { 0xffff,0x0025 },{ 0xffff,0x0026 },{ 0xffff,0x0001 },{ 0xffff,0x0002 },
    { 0xffff,0x0003 },{ 0xffff,0x002a },{ 0xffff,0x0004 },{ 0xffff,0x0005 },
    { 0xffff,0x0006 },{ 0xffff,0x0007 },{ 0xffff,0x0008 },{ 0xffff,0x005b },
    { 0xffff,0x005c },{ 0xffff,0x005d },{ 0xffff,0x005e },{ 0xffff,0x005f },
    { 0xffff,0x0060 },{ 0xffff,0x0009 },{ 0xffff,0x000a },{ 0xffff,0x000b },
    { 0xffff,0x000c },{ 0xffff,0x000d },{ 0xffff,0x000e },{ 0xffff,0x000f },
    { 0xffff,0x0010 },{ 0xffff,0x0011 },{ 0xffff,0x0012 },{ 0xffff,0x0013 },
    { 0xffff,0x003b },{ 0xffff,0x003c },{ 0xffff,0x0014 },{ 0xffff,0x003e },
    { 0xffff,0x0015 },{ 0xffff,0x0040 },{ 0xffff,0x0016 },{ 0xffff,0x0017 },
    { 0xffff,0x0018 },{ 0xffff,0x0019 },{ 0xffff,0x001a },{ 0xffff,0x001b },
    { 0xffff,0x001c },{ 0xffff,0x001d },{ 0xffff,0x001e },{ 0xffff,0x007b },
    { 0xffff,0x007c },{ 0xffff,0x007d },{ 0xffff,0x007e },{ 0xffff,0x007f }
  };

/* ISO 646 basic 1983 (used with VisibleStrings) */

static const SshUInt16 ssh_charset_visible[] =
  {
    0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
    0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
    0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
    0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,0xffff,
    0x0020,0x0021,0x0022,0xffff,0xffff,0x0025,0x0026,0x0027,
    0x0028,0x0029,0x002a,0x002b,0x002c,0x002d,0x002e,0x002f,
    0x0030,0x0031,0x0032,0x0033,0x0034,0x0035,0x0036,0x0037,
    0x0038,0x0039,0x003a,0x003b,0x003c,0x003d,0x003e,0x003f,
    0xffff,0x0041,0x0042,0x0043,0x0044,0x0045,0x0046,0x0047,
    0x0048,0x0049,0x004a,0x004b,0x004c,0x004d,0x004e,0x004f,
    0x0050,0x0051,0x0052,0x0053,0x0054,0x0055,0x0056,0x0057,
    0x0058,0x0059,0x005a,0xffff,0xffff,0xffff,0xffff,0x005f,
    0xffff,0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,
    0x0068,0x0069,0x006a,0x006b,0x006c,0x006d,0x006e,0x006f,
    0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,
    0x0078,0x0079,0x007a,0xffff,0xffff,0xffff,0xffff,0xffff
  };

static const SshCharsetMap ssh_charset_visible_map[] =
  {
    { 0x0020,0x0020 },{ 0x0021,0x0021 },{ 0x0022,0x0022 },{ 0x0025,0x0025 },
    { 0x0026,0x0026 },{ 0x0027,0x0027 },{ 0x0028,0x0028 },{ 0x0029,0x0029 },
    { 0x002a,0x002a },{ 0x002b,0x002b },{ 0x002c,0x002c },{ 0x002d,0x002d },
    { 0x002e,0x002e },{ 0x002f,0x002f },{ 0x0030,0x0030 },{ 0x0031,0x0031 },
    { 0x0032,0x0032 },{ 0x0033,0x0033 },{ 0x0034,0x0034 },{ 0x0035,0x0035 },
    { 0x0036,0x0036 },{ 0x0037,0x0037 },{ 0x0038,0x0038 },{ 0x0039,0x0039 },
    { 0x003a,0x003a },{ 0x003b,0x003b },{ 0x003c,0x003c },{ 0x003d,0x003d },
    { 0x003e,0x003e },{ 0x003f,0x003f },{ 0x0041,0x0041 },{ 0x0042,0x0042 },
    { 0x0043,0x0043 },{ 0x0044,0x0044 },{ 0x0045,0x0045 },{ 0x0046,0x0046 },
    { 0x0047,0x0047 },{ 0x0048,0x0048 },{ 0x0049,0x0049 },{ 0x004a,0x004a },
    { 0x004b,0x004b },{ 0x004c,0x004c },{ 0x004d,0x004d },{ 0x004e,0x004e },
    { 0x004f,0x004f },{ 0x0050,0x0050 },{ 0x0051,0x0051 },{ 0x0052,0x0052 },
    { 0x0053,0x0053 },{ 0x0054,0x0054 },{ 0x0055,0x0055 },{ 0x0056,0x0056 },
    { 0x0057,0x0057 },{ 0x0058,0x0058 },{ 0x0059,0x0059 },{ 0x005a,0x005a },
    { 0x005f,0x005f },{ 0x0061,0x0061 },{ 0x0062,0x0062 },{ 0x0063,0x0063 },
    { 0x0064,0x0064 },{ 0x0065,0x0065 },{ 0x0066,0x0066 },{ 0x0067,0x0067 },
    { 0x0068,0x0068 },{ 0x0069,0x0069 },{ 0x006a,0x006a },{ 0x006b,0x006b },
    { 0x006c,0x006c },{ 0x006d,0x006d },{ 0x006e,0x006e },{ 0x006f,0x006f },
    { 0x0070,0x0070 },{ 0x0071,0x0071 },{ 0x0072,0x0072 },{ 0x0073,0x0073 },
    { 0x0074,0x0074 },{ 0x0075,0x0075 },{ 0x0076,0x0076 },{ 0x0077,0x0077 },
    { 0x0078,0x0078 },{ 0x0079,0x0079 },{ 0x007a,0x007a },{ 0xffff,0x000e },
    { 0xffff,0x000f },{ 0xffff,0x0010 },{ 0xffff,0x0011 },{ 0xffff,0x0012 },
    { 0xffff,0x0013 },{ 0xffff,0x0014 },{ 0xffff,0x0015 },{ 0xffff,0x005b },
    { 0xffff,0x005c },{ 0xffff,0x005d },{ 0xffff,0x005e },{ 0xffff,0x0016 },
    { 0xffff,0x0060 },{ 0xffff,0x0017 },{ 0xffff,0x0018 },{ 0xffff,0x0019 },
    { 0xffff,0x001a },{ 0xffff,0x001b },{ 0xffff,0x001c },{ 0xffff,0x001d },
    { 0xffff,0x0040 },{ 0xffff,0x001e },{ 0xffff,0x001f },{ 0xffff,0x0000 },
    { 0xffff,0x0001 },{ 0xffff,0x0002 },{ 0xffff,0x0023 },{ 0xffff,0x0024 },
    { 0xffff,0x0003 },{ 0xffff,0x0004 },{ 0xffff,0x0005 },{ 0xffff,0x0006 },
    { 0xffff,0x0007 },{ 0xffff,0x0008 },{ 0xffff,0x0009 },{ 0xffff,0x000a },
    { 0xffff,0x000b },{ 0xffff,0x000c },{ 0xffff,0x000d },{ 0xffff,0x007b },
    { 0xffff,0x007c },{ 0xffff,0x007d },{ 0xffff,0x007e },{ 0xffff,0x007f }
  };

/* ISO/IEC 6937:1994, T.61 or Teletext character map. */
static const SshCharsetMap ssh_charset_t61_unicode_map[] = {
  {0x0000,0x0000},{0x0001,0x0001},{0x0002,0x0002},{0x0003,0x0003},
  {0x0004,0x0004},{0x0005,0x0005},{0x0006,0x0006},{0x0007,0x0007},
  {0x0008,0x0008},{0x0009,0x0009},{0x000a,0x000a},{0x000b,0x000b},
  {0x000c,0x000c},{0x000d,0x000d},{0x000e,0x000e},{0x000f,0x000f},
  {0x0010,0x0010},{0x0011,0x0011},{0x0012,0x0012},{0x0013,0x0013},
  {0x0014,0x0014},{0x0015,0x0015},{0x0016,0x0016},{0x0017,0x0017},
  {0x0018,0x0018},{0x0019,0x0019},{0x001a,0x001a},{0x001b,0x001b},
  {0x001c,0x001c},{0x001d,0x001d},{0x001e,0x001e},{0x001f,0x001f},
  {0x0020,0x0020},{0x0021,0x0021},{0x0022,0x0022},{0x0023,0x0023},
  {0x0024,0x00a4},{0x0025,0x0025},{0x0026,0x0026},{0x0027,0x0027},
  {0x0028,0x0028},{0x0029,0x0029},{0x002a,0x002a},{0x002b,0x002b},
  {0x002c,0x002c},{0x002d,0x002d},{0x002e,0x002e},{0x002f,0x002f},
  {0x0030,0x0030},{0x0031,0x0031},{0x0032,0x0032},{0x0033,0x0033},
  {0x0034,0x0034},{0x0035,0x0035},{0x0036,0x0036},{0x0037,0x0037},
  {0x0038,0x0038},{0x0039,0x0039},{0x003a,0x003a},{0x003b,0x003b},
  {0x003c,0x003c},{0x003d,0x003d},{0x003e,0x003e},{0x003f,0x003f},
  {0x0040,0x0040},{0x0041,0x0041},{0x0042,0x0042},{0x0043,0x0043},
  {0x0044,0x0044},{0x0045,0x0045},{0x0046,0x0046},{0x0047,0x0047},
  {0x0048,0x0048},{0x0049,0x0049},{0x004a,0x004a},{0x004b,0x004b},
  {0x004c,0x004c},{0x004d,0x004d},{0x004e,0x004e},{0x004f,0x004f},
  {0x0050,0x0050},{0x0051,0x0051},{0x0052,0x0052},{0x0053,0x0053},
  {0x0054,0x0054},{0x0055,0x0055},{0x0056,0x0056},{0x0057,0x0057},
  {0x0058,0x0058},{0x0059,0x0059},{0x005a,0x005a},{0x005b,0x005b},
  {0x005c,0x005c},{0x005d,0x005d},{0x005e,0x005e},{0x005f,0x005f},
  {0x0060,0x0060},{0x0061,0x0061},{0x0062,0x0062},{0x0063,0x0063},
  {0x0064,0x0064},{0x0065,0x0065},{0x0066,0x0066},{0x0067,0x0067},
  {0x0068,0x0068},{0x0069,0x0069},{0x006a,0x006a},{0x006b,0x006b},
  {0x006c,0x006c},{0x006d,0x006d},{0x006e,0x006e},{0x006f,0x006f},
  {0x0070,0x0070},{0x0071,0x0071},{0x0072,0x0072},{0x0073,0x0073},
  {0x0074,0x0074},{0x0075,0x0075},{0x0076,0x0076},{0x0077,0x0077},
  {0x0078,0x0078},{0x0079,0x0079},{0x007a,0x007a},{0x007b,0x007b},
  {0x007c,0x007c},{0x007d,0x007d},{0x007e,0x007e},{0x007f,0x007f},
  {0x0080,0x0080},{0x0081,0x0081},{0x0082,0x0082},{0x0083,0x0083},
  {0x0084,0x0084},{0x0085,0x0085},{0x0086,0x0086},{0x0087,0x0087},
  {0x0088,0x0088},{0x0089,0x0089},{0x008a,0x008a},{0x008b,0x008b},
  {0x008c,0x008c},{0x008d,0x008d},{0x008e,0x008e},{0x008f,0x008f},
  {0x0090,0x0090},{0x0091,0x0091},{0x0092,0x0092},{0x0093,0x0093},
  {0x0094,0x0094},{0x0095,0x0095},{0x0096,0x0096},{0x0097,0x0097},
  {0x0098,0x0098},{0x0099,0x0099},{0x009a,0x009a},{0x009b,0x009b},
  {0x009c,0x009c},{0x009d,0x009d},{0x009e,0x009e},{0x009f,0x009f},
  {0x00a0,0x00a0},{0x00a1,0x00a1},{0x00a2,0x00a2},{0x00a3,0x00a3},
  {0x00a4,0x0024},{0x00a5,0x00a5},{0x00a7,0x00a7},{0x00a9,0x2018},
  {0x00aa,0x201c},{0x00ab,0x00ab},{0x00ac,0x2190},{0x00ad,0x2191},
  {0x00ae,0x2192},{0x00af,0x2193},{0x00b0,0x00b0},{0x00b1,0x00b1},
  {0x00b2,0x00b2},{0x00b3,0x00b3},{0x00b4,0x00d7},{0x00b5,0x00b5},
  {0x00b6,0x00b6},{0x00b7,0x00b7},{0x00b8,0x00f7},{0x00b9,0x2019},
  {0x00ba,0x201d},{0x00bb,0x00bb},{0x00bc,0x00bc},{0x00bd,0x00bd},
  {0x00be,0x00be},{0x00bf,0x00bf},{0x00c1,0xe002},{0x00c2,0xe003},
  {0x00c3,0xe004},{0x00c4,0xe005},{0x00c5,0xe006},{0x00c6,0xe007},
  {0x00c7,0xe008},{0x00c8,0xe009},{0x00ca,0xe00a},{0x00cb,0xe00b},
  {0x00cc,0xe00c},{0x00cd,0xe00d},{0x00ce,0xe00e},{0x00cf,0xe00f},
  {0x00d0,0x2014},{0x00d1,0x00b9},{0x00d2,0x00ae},{0x00d3,0x00a9},
  {0x00d4,0x2122},{0x00d5,0x266a},{0x00d6,0x00ac},{0x00d7,0x00a6},
  {0x00dc,0x215b},{0x00dd,0x215c},{0x00de,0x215d},{0x00df,0x215e},
  {0x00e0,0x2126},{0x00e1,0x00c6},{0x00e2,0x00d0},{0x00e3,0x00aa},
  {0x00e4,0x0126},{0x00e6,0x0132},{0x00e7,0x013f},{0x00e8,0x0141},
  {0x00e9,0x00d8},{0x00ea,0x0152},{0x00eb,0x00ba},{0x00ec,0x00de},
  {0x00ed,0x0166},{0x00ee,0x014a},{0x00ef,0x0149},{0x00f0,0x0138},
  {0x00f1,0x00e6},{0x00f2,0x0111},{0x00f3,0x00f0},{0x00f4,0x0127},
  {0x00f5,0x0131},{0x00f6,0x0133},{0x00f7,0x0140},{0x00f8,0x0142},
  {0x00f9,0x00f8},{0x00fa,0x0153},{0x00fb,0x00df},{0x00fc,0x00fe},
  {0x00fd,0x0167},{0x00fe,0x014b},{0x00ff,0x00ad},{0xc141,0x00c0},
  {0xc145,0x00c8},{0xc149,0x00cc},{0xc14f,0x00d2},{0xc155,0x00d9},
  {0xc161,0x00e0},{0xc165,0x00e8},{0xc169,0x00ec},{0xc16f,0x00f2},
  {0xc175,0x00f9},{0xc220,0x00b4},{0xc241,0x00c1},{0xc243,0x0106},
  {0xc245,0x00c9},{0xc249,0x00cd},{0xc24c,0x0139},{0xc24e,0x0143},
  {0xc24f,0x00d3},{0xc252,0x0154},{0xc253,0x015a},{0xc255,0x00da},
  {0xc259,0x00dd},{0xc25a,0x0179},{0xc261,0x00e1},{0xc263,0x0107},
  {0xc265,0x00e9},{0xc269,0x00ed},{0xc26c,0x013a},{0xc26e,0x0144},
  {0xc26f,0x00f3},{0xc272,0x0155},{0xc273,0x015b},{0xc275,0x00fa},
  {0xc279,0x00fd},{0xc27a,0x017a},{0xc341,0x00c2},{0xc343,0x0108},
  {0xc345,0x00ca},{0xc347,0x011c},{0xc348,0x0124},{0xc349,0x00ce},
  {0xc34a,0x0134},{0xc34f,0x00d4},{0xc353,0x015c},{0xc355,0x00db},
  {0xc357,0x0174},{0xc359,0x0176},{0xc361,0x00e2},{0xc363,0x0109},
  {0xc365,0x00ea},{0xc367,0x011d},{0xc368,0x0125},{0xc369,0x00ee},
  {0xc36a,0x0135},{0xc36f,0x00f4},{0xc373,0x015d},{0xc375,0x00fb},
  {0xc377,0x0175},{0xc379,0x0177},{0xc441,0x00c3},{0xc449,0x0128},
  {0xc44e,0x00d1},{0xc44f,0x00d5},{0xc455,0x0168},{0xc461,0x00e3},
  {0xc469,0x0129},{0xc46e,0x00f1},{0xc46f,0x00f5},{0xc475,0x0169},
  {0xc520,0x00af},{0xc541,0x0100},{0xc545,0x0112},{0xc549,0x012a},
  {0xc54f,0x014c},{0xc555,0x016a},{0xc561,0x0101},{0xc565,0x0113},
  {0xc569,0x012b},{0xc56f,0x014d},{0xc575,0x016b},{0xc620,0x02d8},
  {0xc641,0x0102},{0xc647,0x011e},{0xc655,0x016c},{0xc661,0x0103},
  {0xc667,0x011f},{0xc675,0x016d},{0xc720,0x02d9},{0xc743,0x010a},
  {0xc745,0x0116},{0xc747,0x0120},{0xc749,0x0130},{0xc75a,0x017b},
  {0xc763,0x010b},{0xc765,0x0117},{0xc767,0x0121},{0xc77a,0x017c},
  {0xc820,0x00a8},{0xc841,0x00c4},{0xc845,0x00cb},{0xc849,0x00cf},
  {0xc84f,0x00d6},{0xc855,0x00dc},{0xc859,0x0178},{0xc861,0x00e4},
  {0xc865,0x00eb},{0xc869,0x00ef},{0xc86f,0x00f6},{0xc875,0x00fc},
  {0xc879,0x00ff},{0xca20,0x02da},{0xca41,0x00c5},{0xca55,0x016e},
  {0xca61,0x00e5},{0xca75,0x016f},{0xcb20,0x00b8},{0xcb43,0x00c7},
  {0xcb47,0x0122},{0xcb4b,0x0136},{0xcb4c,0x013b},{0xcb4e,0x0145},
  {0xcb52,0x0156},{0xcb53,0x015e},{0xcb54,0x0162},{0xcb63,0x00e7},
  {0xcb67,0x0123},{0xcb6b,0x0137},{0xcb6c,0x013c},{0xcb6e,0x0146},
  {0xcb72,0x0157},{0xcb73,0x015f},{0xcb74,0x0163},{0xcd20,0x02dd},
  {0xcd4f,0x0150},{0xcd55,0x0170},{0xcd6f,0x0151},{0xcd75,0x0171},
  {0xce20,0x02db},{0xce41,0x0104},{0xce45,0x0118},{0xce49,0x012e},
  {0xce55,0x0172},{0xce61,0x0105},{0xce65,0x0119},{0xce69,0x012f},
  {0xce75,0x0173},{0xcf20,0x02c7},{0xcf43,0x010c},{0xcf44,0x010e},
  {0xcf45,0x011a},{0xcf4c,0x013d},{0xcf4e,0x0147},{0xcf52,0x0158},
  {0xcf53,0x0160},{0xcf54,0x0164},{0xcf5a,0x017d},{0xcf63,0x010d},
  {0xcf64,0x010f},{0xcf65,0x011b},{0xcf6c,0x013e},{0xcf6e,0x0148},
  {0xcf72,0x0159},{0xcf73,0x0161},{0xcf74,0x0165},{0xcf7a,0x017e},
  {0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},
  {0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},
  {0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},
  {0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},
  {0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},
  {0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},
  {0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},
  {0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},
  {0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},
  {0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},
  {0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},
  {0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},
  {0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},
  {0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},
  {0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},
  {0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},
  {0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},
  {0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},
  {0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},
  {0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},
  {0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},
  {0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},
  {0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},
  {0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},
  {0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},
};

static const SshCharsetMap ssh_charset_t61_map[] = {
  {0x0000,0x0000},{0x0001,0x0001},{0x0002,0x0002},{0x0003,0x0003},
  {0x0004,0x0004},{0x0005,0x0005},{0x0006,0x0006},{0x0007,0x0007},
  {0x0008,0x0008},{0x0009,0x0009},{0x000a,0x000a},{0x000b,0x000b},
  {0x000c,0x000c},{0x000d,0x000d},{0x000e,0x000e},{0x000f,0x000f},
  {0x0010,0x0010},{0x0011,0x0011},{0x0012,0x0012},{0x0013,0x0013},
  {0x0014,0x0014},{0x0015,0x0015},{0x0016,0x0016},{0x0017,0x0017},
  {0x0018,0x0018},{0x0019,0x0019},{0x001a,0x001a},{0x001b,0x001b},
  {0x001c,0x001c},{0x001d,0x001d},{0x001e,0x001e},{0x001f,0x001f},
  {0x0020,0x0020},{0x0021,0x0021},{0x0022,0x0022},{0x0023,0x0023},
  {0x0024,0x00a4},{0x0025,0x0025},{0x0026,0x0026},{0x0027,0x0027},
  {0x0028,0x0028},{0x0029,0x0029},{0x002a,0x002a},{0x002b,0x002b},
  {0x002c,0x002c},{0x002d,0x002d},{0x002e,0x002e},{0x002f,0x002f},
  {0x0030,0x0030},{0x0031,0x0031},{0x0032,0x0032},{0x0033,0x0033},
  {0x0034,0x0034},{0x0035,0x0035},{0x0036,0x0036},{0x0037,0x0037},
  {0x0038,0x0038},{0x0039,0x0039},{0x003a,0x003a},{0x003b,0x003b},
  {0x003c,0x003c},{0x003d,0x003d},{0x003e,0x003e},{0x003f,0x003f},
  {0x0040,0x0040},{0x0041,0x0041},{0x0042,0x0042},{0x0043,0x0043},
  {0x0044,0x0044},{0x0045,0x0045},{0x0046,0x0046},{0x0047,0x0047},
  {0x0048,0x0048},{0x0049,0x0049},{0x004a,0x004a},{0x004b,0x004b},
  {0x004c,0x004c},{0x004d,0x004d},{0x004e,0x004e},{0x004f,0x004f},
  {0x0050,0x0050},{0x0051,0x0051},{0x0052,0x0052},{0x0053,0x0053},
  {0x0054,0x0054},{0x0055,0x0055},{0x0056,0x0056},{0x0057,0x0057},
  {0x0058,0x0058},{0x0059,0x0059},{0x005a,0x005a},{0x005b,0x005b},
  {0x005c,0x005c},{0x005d,0x005d},{0x005e,0x005e},{0x005f,0x005f},
  {0x0060,0x0060},{0x0061,0x0061},{0x0062,0x0062},{0x0063,0x0063},
  {0x0064,0x0064},{0x0065,0x0065},{0x0066,0x0066},{0x0067,0x0067},
  {0x0068,0x0068},{0x0069,0x0069},{0x006a,0x006a},{0x006b,0x006b},
  {0x006c,0x006c},{0x006d,0x006d},{0x006e,0x006e},{0x006f,0x006f},
  {0x0070,0x0070},{0x0071,0x0071},{0x0072,0x0072},{0x0073,0x0073},
  {0x0074,0x0074},{0x0075,0x0075},{0x0076,0x0076},{0x0077,0x0077},
  {0x0078,0x0078},{0x0079,0x0079},{0x007a,0x007a},{0x007b,0x007b},
  {0x007c,0x007c},{0x007d,0x007d},{0x007e,0x007e},{0x007f,0x007f},
  {0x0080,0x0080},{0x0081,0x0081},{0x0082,0x0082},{0x0083,0x0083},
  {0x0084,0x0084},{0x0085,0x0085},{0x0086,0x0086},{0x0087,0x0087},
  {0x0088,0x0088},{0x0089,0x0089},{0x008a,0x008a},{0x008b,0x008b},
  {0x008c,0x008c},{0x008d,0x008d},{0x008e,0x008e},{0x008f,0x008f},
  {0x0090,0x0090},{0x0091,0x0091},{0x0092,0x0092},{0x0093,0x0093},
  {0x0094,0x0094},{0x0095,0x0095},{0x0096,0x0096},{0x0097,0x0097},
  {0x0098,0x0098},{0x0099,0x0099},{0x009a,0x009a},{0x009b,0x009b},
  {0x009c,0x009c},{0x009d,0x009d},{0x009e,0x009e},{0x009f,0x009f},
  {0x00a0,0x00a0},{0x00a1,0x00a1},{0x00a2,0x00a2},{0x00a3,0x00a3},
  {0x00a4,0x0024},{0x00a5,0x00a5},{0x00a6,0x00d7},{0x00a7,0x00a7},
  {0x00a8,0xc820},{0x00a9,0x00d3},{0x00aa,0x00e3},{0x00ab,0x00ab},
  {0x00ac,0x00d6},{0x00ad,0x00ff},{0x00ae,0x00d2},{0x00af,0xc520},
  {0x00b0,0x00b0},{0x00b1,0x00b1},{0x00b2,0x00b2},{0x00b3,0x00b3},
  {0x00b4,0xc220},{0x00b5,0x00b5},{0x00b6,0x00b6},{0x00b7,0x00b7},
  {0x00b8,0xcb20},{0x00b9,0x00d1},{0x00ba,0x00eb},{0x00bb,0x00bb},
  {0x00bc,0x00bc},{0x00bd,0x00bd},{0x00be,0x00be},{0x00bf,0x00bf},
  {0x00c0,0xc141},{0x00c1,0xc241},{0x00c2,0xc341},{0x00c3,0xc441},
  {0x00c4,0xc841},{0x00c5,0xca41},{0x00c6,0x00e1},{0x00c7,0xcb43},
  {0x00c8,0xc145},{0x00c9,0xc245},{0x00ca,0xc345},{0x00cb,0xc845},
  {0x00cc,0xc149},{0x00cd,0xc249},{0x00ce,0xc349},{0x00cf,0xc849},
  {0x00d0,0x00e2},{0x00d1,0xc44e},{0x00d2,0xc14f},{0x00d3,0xc24f},
  {0x00d4,0xc34f},{0x00d5,0xc44f},{0x00d6,0xc84f},{0x00d7,0x00b4},
  {0x00d8,0x00e9},{0x00d9,0xc155},{0x00da,0xc255},{0x00db,0xc355},
  {0x00dc,0xc855},{0x00dd,0xc259},{0x00de,0x00ec},{0x00df,0x00fb},
  {0x00e0,0xc161},{0x00e1,0xc261},{0x00e2,0xc361},{0x00e3,0xc461},
  {0x00e4,0xc861},{0x00e5,0xca61},{0x00e6,0x00f1},{0x00e7,0xcb63},
  {0x00e8,0xc165},{0x00e9,0xc265},{0x00ea,0xc365},{0x00eb,0xc865},
  {0x00ec,0xc169},{0x00ed,0xc269},{0x00ee,0xc369},{0x00ef,0xc869},
  {0x00f0,0x00f3},{0x00f1,0xc46e},{0x00f2,0xc16f},{0x00f3,0xc26f},
  {0x00f4,0xc36f},{0x00f5,0xc46f},{0x00f6,0xc86f},{0x00f7,0x00b8},
  {0x00f8,0x00f9},{0x00f9,0xc175},{0x00fa,0xc275},{0x00fb,0xc375},
  {0x00fc,0xc875},{0x00fd,0xc279},{0x00fe,0x00fc},{0x00ff,0xc879},
  {0x0100,0xc541},{0x0101,0xc561},{0x0102,0xc641},{0x0103,0xc661},
  {0x0104,0xce41},{0x0105,0xce61},{0x0106,0xc243},{0x0107,0xc263},
  {0x0108,0xc343},{0x0109,0xc363},{0x010a,0xc743},{0x010b,0xc763},
  {0x010c,0xcf43},{0x010d,0xcf63},{0x010e,0xcf44},{0x010f,0xcf64},
  {0x0111,0x00f2},{0x0112,0xc545},{0x0113,0xc565},{0x0116,0xc745},
  {0x0117,0xc765},{0x0118,0xce45},{0x0119,0xce65},{0x011a,0xcf45},
  {0x011b,0xcf65},{0x011c,0xc347},{0x011d,0xc367},{0x011e,0xc647},
  {0x011f,0xc667},{0x0120,0xc747},{0x0121,0xc767},{0x0122,0xcb47},
  {0x0123,0xcb67},{0x0124,0xc348},{0x0125,0xc368},{0x0126,0x00e4},
  {0x0127,0x00f4},{0x0128,0xc449},{0x0129,0xc469},{0x012a,0xc549},
  {0x012b,0xc569},{0x012e,0xce49},{0x012f,0xce69},{0x0130,0xc749},
  {0x0131,0x00f5},{0x0132,0x00e6},{0x0133,0x00f6},{0x0134,0xc34a},
  {0x0135,0xc36a},{0x0136,0xcb4b},{0x0137,0xcb6b},{0x0138,0x00f0},
  {0x0139,0xc24c},{0x013a,0xc26c},{0x013b,0xcb4c},{0x013c,0xcb6c},
  {0x013d,0xcf4c},{0x013e,0xcf6c},{0x013f,0x00e7},{0x0140,0x00f7},
  {0x0141,0x00e8},{0x0142,0x00f8},{0x0143,0xc24e},{0x0144,0xc26e},
  {0x0145,0xcb4e},{0x0146,0xcb6e},{0x0147,0xcf4e},{0x0148,0xcf6e},
  {0x0149,0x00ef},{0x014a,0x00ee},{0x014b,0x00fe},{0x014c,0xc54f},
  {0x014d,0xc56f},{0x0150,0xcd4f},{0x0151,0xcd6f},{0x0152,0x00ea},
  {0x0153,0x00fa},{0x0154,0xc252},{0x0155,0xc272},{0x0156,0xcb52},
  {0x0157,0xcb72},{0x0158,0xcf52},{0x0159,0xcf72},{0x015a,0xc253},
  {0x015b,0xc273},{0x015c,0xc353},{0x015d,0xc373},{0x015e,0xcb53},
  {0x015f,0xcb73},{0x0160,0xcf53},{0x0161,0xcf73},{0x0162,0xcb54},
  {0x0163,0xcb74},{0x0164,0xcf54},{0x0165,0xcf74},{0x0166,0x00ed},
  {0x0167,0x00fd},{0x0168,0xc455},{0x0169,0xc475},{0x016a,0xc555},
  {0x016b,0xc575},{0x016c,0xc655},{0x016d,0xc675},{0x016e,0xca55},
  {0x016f,0xca75},{0x0170,0xcd55},{0x0171,0xcd75},{0x0172,0xce55},
  {0x0173,0xce75},{0x0174,0xc357},{0x0175,0xc377},{0x0176,0xc359},
  {0x0177,0xc379},{0x0178,0xc859},{0x0179,0xc25a},{0x017a,0xc27a},
  {0x017b,0xc75a},{0x017c,0xc77a},{0x017d,0xcf5a},{0x017e,0xcf7a},
  {0x02c7,0xcf20},{0x02d8,0xc620},{0x02d9,0xc720},{0x02da,0xca20},
  {0x02db,0xce20},{0x02dd,0xcd20},{0x2014,0x00d0},{0x2018,0x00a9},
  {0x2019,0x00b9},{0x201c,0x00aa},{0x201d,0x00ba},{0x2122,0x00d4},
  {0x2126,0x00e0},{0x215b,0x00dc},{0x215c,0x00dd},{0x215d,0x00de},
  {0x215e,0x00df},{0x2190,0x00ac},{0x2191,0x00ad},{0x2192,0x00ae},
  {0x2193,0x00af},{0x266a,0x00d5},{0xe002,0x00c1},{0xe003,0x00c2},
  {0xe004,0x00c3},{0xe005,0x00c4},{0xe006,0x00c5},{0xe007,0x00c6},
  {0xe008,0x00c7},{0xe009,0x00c8},{0xe00a,0x00ca},{0xe00b,0x00cb},
  {0xe00c,0x00cc},{0xe00d,0x00cd},{0xe00e,0x00ce},{0xe00f,0x00cf},
  {0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},
  {0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},
  {0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},
  {0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},
  {0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},
  {0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},
  {0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},
  {0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},
  {0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},
  {0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},
  {0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},
  {0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},
  {0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},
  {0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},
  {0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},
  {0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},
  {0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},
  {0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},
  {0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},
  {0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},
  {0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},
  {0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},
  {0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},
  {0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},
  {0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},{0xffff,0xffff},
};

/* ISO 8859-2:1987 character set. */
static const SshUInt16 ssh_charset_iso_8859_2[] =
  {
    0x0000,0x0001,0x0002,0x0003,0x0004,0x0005,0x0006,0x0007,
    0x0008,0x0009,0x000a,0x000b,0x000c,0x000d,0x000e,0x000f,
    0x0010,0x0011,0x0012,0x0013,0x0014,0x0015,0x0016,0x0017,
    0x0018,0x0019,0x001a,0x001b,0x001c,0x001d,0x001e,0x001f,
    0x0020,0x0021,0x0022,0x0023,0x0024,0x0025,0x0026,0x0027,
    0x0028,0x0029,0x002a,0x002b,0x002c,0x002d,0x002e,0x002f,
    0x0030,0x0031,0x0032,0x0033,0x0034,0x0035,0x0036,0x0037,
    0x0038,0x0039,0x003a,0x003b,0x003c,0x003d,0x003e,0x003f,
    0x0040,0x0041,0x0042,0x0043,0x0044,0x0045,0x0046,0x0047,
    0x0048,0x0049,0x004a,0x004b,0x004c,0x004d,0x004e,0x004f,
    0x0050,0x0051,0x0052,0x0053,0x0054,0x0055,0x0056,0x0057,
    0x0058,0x0059,0x005a,0x005b,0x005c,0x005d,0x005e,0x005f,
    0x0060,0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,
    0x0068,0x0069,0x006a,0x006b,0x006c,0x006d,0x006e,0x006f,
    0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,
    0x0078,0x0079,0x007a,0x007b,0x007c,0x007d,0x007e,0x007f,
    0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,
    0x0088,0x0089,0x008a,0x008b,0x008c,0x008d,0x008e,0x008f,
    0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,
    0x0098,0x0099,0x009a,0x009b,0x009c,0x009d,0x009e,0x009f,
    0x00a0,0x0104,0x02d8,0x0141,0x00a4,0x013d,0x015a,0x00a7,
    0x00a8,0x0160,0x015e,0x0164,0x0179,0x00ad,0x017d,0x017b,
    0x00b0,0x0105,0x02db,0x0142,0x00b4,0x013e,0x015b,0x02c7,
    0x00b8,0x0161,0x015f,0x0165,0x017a,0x02dd,0x017e,0x017c,
    0x0154,0x00c1,0x00c2,0x0102,0x00c4,0x0139,0x0106,0x00c7,
    0x010c,0x00c9,0x0118,0x00cb,0x011a,0x00cd,0x00ce,0x010e,
    0x0110,0x0143,0x0147,0x00d3,0x00d4,0x0150,0x00d6,0x00d7,
    0x0158,0x016e,0x00da,0x0170,0x00dc,0x00dd,0x0162,0x00df,
    0x0155,0x00e1,0x00e2,0x0103,0x00e4,0x013a,0x0107,0x00e7,
    0x010d,0x00e9,0x0119,0x00eb,0x011b,0x00ed,0x00ee,0x010f,
    0x0111,0x0144,0x0148,0x00f3,0x00f4,0x0151,0x00f6,0x00f7,
    0x0159,0x016f,0x00fa,0x0171,0x00fc,0x00fd,0x0163,0x02d9
  };

static const SshCharsetMap ssh_charset_iso_8859_2_map[] =
  {
    { 0x0000,0x0000 },{ 0x0001,0x0001 },{ 0x0002,0x0002 },{ 0x0003,0x0003 },
    { 0x0004,0x0004 },{ 0x0005,0x0005 },{ 0x0006,0x0006 },{ 0x0007,0x0007 },
    { 0x0008,0x0008 },{ 0x0009,0x0009 },{ 0x000a,0x000a },{ 0x000b,0x000b },
    { 0x000c,0x000c },{ 0x000d,0x000d },{ 0x000e,0x000e },{ 0x000f,0x000f },
    { 0x0010,0x0010 },{ 0x0011,0x0011 },{ 0x0012,0x0012 },{ 0x0013,0x0013 },
    { 0x0014,0x0014 },{ 0x0015,0x0015 },{ 0x0016,0x0016 },{ 0x0017,0x0017 },
    { 0x0018,0x0018 },{ 0x0019,0x0019 },{ 0x001a,0x001a },{ 0x001b,0x001b },
    { 0x001c,0x001c },{ 0x001d,0x001d },{ 0x001e,0x001e },{ 0x001f,0x001f },
    { 0x0020,0x0020 },{ 0x0021,0x0021 },{ 0x0022,0x0022 },{ 0x0023,0x0023 },
    { 0x0024,0x0024 },{ 0x0025,0x0025 },{ 0x0026,0x0026 },{ 0x0027,0x0027 },
    { 0x0028,0x0028 },{ 0x0029,0x0029 },{ 0x002a,0x002a },{ 0x002b,0x002b },
    { 0x002c,0x002c },{ 0x002d,0x002d },{ 0x002e,0x002e },{ 0x002f,0x002f },
    { 0x0030,0x0030 },{ 0x0031,0x0031 },{ 0x0032,0x0032 },{ 0x0033,0x0033 },
    { 0x0034,0x0034 },{ 0x0035,0x0035 },{ 0x0036,0x0036 },{ 0x0037,0x0037 },
    { 0x0038,0x0038 },{ 0x0039,0x0039 },{ 0x003a,0x003a },{ 0x003b,0x003b },
    { 0x003c,0x003c },{ 0x003d,0x003d },{ 0x003e,0x003e },{ 0x003f,0x003f },
    { 0x0040,0x0040 },{ 0x0041,0x0041 },{ 0x0042,0x0042 },{ 0x0043,0x0043 },
    { 0x0044,0x0044 },{ 0x0045,0x0045 },{ 0x0046,0x0046 },{ 0x0047,0x0047 },
    { 0x0048,0x0048 },{ 0x0049,0x0049 },{ 0x004a,0x004a },{ 0x004b,0x004b },
    { 0x004c,0x004c },{ 0x004d,0x004d },{ 0x004e,0x004e },{ 0x004f,0x004f },
    { 0x0050,0x0050 },{ 0x0051,0x0051 },{ 0x0052,0x0052 },{ 0x0053,0x0053 },
    { 0x0054,0x0054 },{ 0x0055,0x0055 },{ 0x0056,0x0056 },{ 0x0057,0x0057 },
    { 0x0058,0x0058 },{ 0x0059,0x0059 },{ 0x005a,0x005a },{ 0x005b,0x005b },
    { 0x005c,0x005c },{ 0x005d,0x005d },{ 0x005e,0x005e },{ 0x005f,0x005f },
    { 0x0060,0x0060 },{ 0x0061,0x0061 },{ 0x0062,0x0062 },{ 0x0063,0x0063 },
    { 0x0064,0x0064 },{ 0x0065,0x0065 },{ 0x0066,0x0066 },{ 0x0067,0x0067 },
    { 0x0068,0x0068 },{ 0x0069,0x0069 },{ 0x006a,0x006a },{ 0x006b,0x006b },
    { 0x006c,0x006c },{ 0x006d,0x006d },{ 0x006e,0x006e },{ 0x006f,0x006f },
    { 0x0070,0x0070 },{ 0x0071,0x0071 },{ 0x0072,0x0072 },{ 0x0073,0x0073 },
    { 0x0074,0x0074 },{ 0x0075,0x0075 },{ 0x0076,0x0076 },{ 0x0077,0x0077 },
    { 0x0078,0x0078 },{ 0x0079,0x0079 },{ 0x007a,0x007a },{ 0x007b,0x007b },
    { 0x007c,0x007c },{ 0x007d,0x007d },{ 0x007e,0x007e },{ 0x007f,0x007f },
    { 0x0080,0x0080 },{ 0x0081,0x0081 },{ 0x0082,0x0082 },{ 0x0083,0x0083 },
    { 0x0084,0x0084 },{ 0x0085,0x0085 },{ 0x0086,0x0086 },{ 0x0087,0x0087 },
    { 0x0088,0x0088 },{ 0x0089,0x0089 },{ 0x008a,0x008a },{ 0x008b,0x008b },
    { 0x008c,0x008c },{ 0x008d,0x008d },{ 0x008e,0x008e },{ 0x008f,0x008f },
    { 0x0090,0x0090 },{ 0x0091,0x0091 },{ 0x0092,0x0092 },{ 0x0093,0x0093 },
    { 0x0094,0x0094 },{ 0x0095,0x0095 },{ 0x0096,0x0096 },{ 0x0097,0x0097 },
    { 0x0098,0x0098 },{ 0x0099,0x0099 },{ 0x009a,0x009a },{ 0x009b,0x009b },
    { 0x009c,0x009c },{ 0x009d,0x009d },{ 0x009e,0x009e },{ 0x009f,0x009f },
    { 0x00a0,0x00a0 },{ 0x00a4,0x00a4 },{ 0x00a7,0x00a7 },{ 0x00a8,0x00a8 },
    { 0x00ad,0x00ad },{ 0x00b0,0x00b0 },{ 0x00b4,0x00b4 },{ 0x00b8,0x00b8 },
    { 0x00c1,0x00c1 },{ 0x00c2,0x00c2 },{ 0x00c4,0x00c4 },{ 0x00c7,0x00c7 },
    { 0x00c9,0x00c9 },{ 0x00cb,0x00cb },{ 0x00cd,0x00cd },{ 0x00ce,0x00ce },
    { 0x00d3,0x00d3 },{ 0x00d4,0x00d4 },{ 0x00d6,0x00d6 },{ 0x00d7,0x00d7 },
    { 0x00da,0x00da },{ 0x00dc,0x00dc },{ 0x00dd,0x00dd },{ 0x00df,0x00df },
    { 0x00e1,0x00e1 },{ 0x00e2,0x00e2 },{ 0x00e4,0x00e4 },{ 0x00e7,0x00e7 },
    { 0x00e9,0x00e9 },{ 0x00eb,0x00eb },{ 0x00ed,0x00ed },{ 0x00ee,0x00ee },
    { 0x00f3,0x00f3 },{ 0x00f4,0x00f4 },{ 0x00f6,0x00f6 },{ 0x00f7,0x00f7 },
    { 0x00fa,0x00fa },{ 0x00fc,0x00fc },{ 0x00fd,0x00fd },{ 0x0102,0x00c3 },
    { 0x0103,0x00e3 },{ 0x0104,0x00a1 },{ 0x0105,0x00b1 },{ 0x0106,0x00c6 },
    { 0x0107,0x00e6 },{ 0x010c,0x00c8 },{ 0x010d,0x00e8 },{ 0x010e,0x00cf },
    { 0x010f,0x00ef },{ 0x0110,0x00d0 },{ 0x0111,0x00f0 },{ 0x0118,0x00ca },
    { 0x0119,0x00ea },{ 0x011a,0x00cc },{ 0x011b,0x00ec },{ 0x0139,0x00c5 },
    { 0x013a,0x00e5 },{ 0x013d,0x00a5 },{ 0x013e,0x00b5 },{ 0x0141,0x00a3 },
    { 0x0142,0x00b3 },{ 0x0143,0x00d1 },{ 0x0144,0x00f1 },{ 0x0147,0x00d2 },
    { 0x0148,0x00f2 },{ 0x0150,0x00d5 },{ 0x0151,0x00f5 },{ 0x0154,0x00c0 },
    { 0x0155,0x00e0 },{ 0x0158,0x00d8 },{ 0x0159,0x00f8 },{ 0x015a,0x00a6 },
    { 0x015b,0x00b6 },{ 0x015e,0x00aa },{ 0x015f,0x00ba },{ 0x0160,0x00a9 },
    { 0x0161,0x00b9 },{ 0x0162,0x00de },{ 0x0163,0x00fe },{ 0x0164,0x00ab },
    { 0x0165,0x00bb },{ 0x016e,0x00d9 },{ 0x016f,0x00f9 },{ 0x0170,0x00db },
    { 0x0171,0x00fb },{ 0x0179,0x00ac },{ 0x017a,0x00bc },{ 0x017b,0x00af },
    { 0x017c,0x00bf },{ 0x017d,0x00ae },{ 0x017e,0x00be },{ 0x02c7,0x00b7 },
    { 0x02d8,0x00a2 },{ 0x02d9,0x00ff },{ 0x02db,0x00b2 },{ 0x02dd,0x00bd }
  };

/* ISO 8859-3:1988 character set. */

static const SshUInt16 ssh_charset_iso_8859_3[] =
  {
    0x0000,0x0001,0x0002,0x0003,0x0004,0x0005,0x0006,0x0007,
    0x0008,0x0009,0x000a,0x000b,0x000c,0x000d,0x000e,0x000f,
    0x0010,0x0011,0x0012,0x0013,0x0014,0x0015,0x0016,0x0017,
    0x0018,0x0019,0x001a,0x001b,0x001c,0x001d,0x001e,0x001f,
    0x0020,0x0021,0x0022,0x0023,0x0024,0x0025,0x0026,0x0027,
    0x0028,0x0029,0x002a,0x002b,0x002c,0x002d,0x002e,0x002f,
    0x0030,0x0031,0x0032,0x0033,0x0034,0x0035,0x0036,0x0037,
    0x0038,0x0039,0x003a,0x003b,0x003c,0x003d,0x003e,0x003f,
    0x0040,0x0041,0x0042,0x0043,0x0044,0x0045,0x0046,0x0047,
    0x0048,0x0049,0x004a,0x004b,0x004c,0x004d,0x004e,0x004f,
    0x0050,0x0051,0x0052,0x0053,0x0054,0x0055,0x0056,0x0057,
    0x0058,0x0059,0x005a,0x005b,0x005c,0x005d,0x005e,0x005f,
    0x0060,0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,
    0x0068,0x0069,0x006a,0x006b,0x006c,0x006d,0x006e,0x006f,
    0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,
    0x0078,0x0079,0x007a,0x007b,0x007c,0x007d,0x007e,0x007f,
    0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,
    0x0088,0x0089,0x008a,0x008b,0x008c,0x008d,0x008e,0x008f,
    0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,
    0x0098,0x0099,0x009a,0x009b,0x009c,0x009d,0x009e,0x009f,
    0x00a0,0x0126,0x02d8,0x00a3,0x00a4,0xffff,0x0124,0x00a7,
    0x00a8,0x0130,0x015e,0x011e,0x0134,0x00ad,0xffff,0x017b,
    0x00b0,0x0127,0x00b2,0x00b3,0x00b4,0x00b5,0x0125,0x00b7,
    0x00b8,0x0131,0x015f,0x011f,0x0135,0x00bd,0xffff,0x017c,
    0x00c0,0x00c1,0x00c2,0xffff,0x00c4,0x010a,0x0108,0x00c7,
    0x00c8,0x00c9,0x00ca,0x00cb,0x00cc,0x00cd,0x00ce,0x00cf,
    0xffff,0x00d1,0x00d2,0x00d3,0x00d4,0x0120,0x00d6,0x00d7,
    0x011c,0x00d9,0x00da,0x00db,0x00dc,0x016c,0x015c,0x00df,
    0x00e0,0x00e1,0x00e2,0xffff,0x00e4,0x010b,0x0109,0x00e7,
    0x00e8,0x00e9,0x00ea,0x00eb,0x00ec,0x00ed,0x00ee,0x00ef,
    0xffff,0x00f1,0x00f2,0x00f3,0x00f4,0x0121,0x00f6,0x00f7,
    0x011d,0x00f9,0x00fa,0x00fb,0x00fc,0x016d,0x015d,0x02d9
  };

static const SshCharsetMap ssh_charset_iso_8859_3_map[] =
  {
    { 0x0000,0x0000 },{ 0x0001,0x0001 },{ 0x0002,0x0002 },{ 0x0003,0x0003 },
    { 0x0004,0x0004 },{ 0x0005,0x0005 },{ 0x0006,0x0006 },{ 0x0007,0x0007 },
    { 0x0008,0x0008 },{ 0x0009,0x0009 },{ 0x000a,0x000a },{ 0x000b,0x000b },
    { 0x000c,0x000c },{ 0x000d,0x000d },{ 0x000e,0x000e },{ 0x000f,0x000f },
    { 0x0010,0x0010 },{ 0x0011,0x0011 },{ 0x0012,0x0012 },{ 0x0013,0x0013 },
    { 0x0014,0x0014 },{ 0x0015,0x0015 },{ 0x0016,0x0016 },{ 0x0017,0x0017 },
    { 0x0018,0x0018 },{ 0x0019,0x0019 },{ 0x001a,0x001a },{ 0x001b,0x001b },
    { 0x001c,0x001c },{ 0x001d,0x001d },{ 0x001e,0x001e },{ 0x001f,0x001f },
    { 0x0020,0x0020 },{ 0x0021,0x0021 },{ 0x0022,0x0022 },{ 0x0023,0x0023 },
    { 0x0024,0x0024 },{ 0x0025,0x0025 },{ 0x0026,0x0026 },{ 0x0027,0x0027 },
    { 0x0028,0x0028 },{ 0x0029,0x0029 },{ 0x002a,0x002a },{ 0x002b,0x002b },
    { 0x002c,0x002c },{ 0x002d,0x002d },{ 0x002e,0x002e },{ 0x002f,0x002f },
    { 0x0030,0x0030 },{ 0x0031,0x0031 },{ 0x0032,0x0032 },{ 0x0033,0x0033 },
    { 0x0034,0x0034 },{ 0x0035,0x0035 },{ 0x0036,0x0036 },{ 0x0037,0x0037 },
    { 0x0038,0x0038 },{ 0x0039,0x0039 },{ 0x003a,0x003a },{ 0x003b,0x003b },
    { 0x003c,0x003c },{ 0x003d,0x003d },{ 0x003e,0x003e },{ 0x003f,0x003f },
    { 0x0040,0x0040 },{ 0x0041,0x0041 },{ 0x0042,0x0042 },{ 0x0043,0x0043 },
    { 0x0044,0x0044 },{ 0x0045,0x0045 },{ 0x0046,0x0046 },{ 0x0047,0x0047 },
    { 0x0048,0x0048 },{ 0x0049,0x0049 },{ 0x004a,0x004a },{ 0x004b,0x004b },
    { 0x004c,0x004c },{ 0x004d,0x004d },{ 0x004e,0x004e },{ 0x004f,0x004f },
    { 0x0050,0x0050 },{ 0x0051,0x0051 },{ 0x0052,0x0052 },{ 0x0053,0x0053 },
    { 0x0054,0x0054 },{ 0x0055,0x0055 },{ 0x0056,0x0056 },{ 0x0057,0x0057 },
    { 0x0058,0x0058 },{ 0x0059,0x0059 },{ 0x005a,0x005a },{ 0x005b,0x005b },
    { 0x005c,0x005c },{ 0x005d,0x005d },{ 0x005e,0x005e },{ 0x005f,0x005f },
    { 0x0060,0x0060 },{ 0x0061,0x0061 },{ 0x0062,0x0062 },{ 0x0063,0x0063 },
    { 0x0064,0x0064 },{ 0x0065,0x0065 },{ 0x0066,0x0066 },{ 0x0067,0x0067 },
    { 0x0068,0x0068 },{ 0x0069,0x0069 },{ 0x006a,0x006a },{ 0x006b,0x006b },
    { 0x006c,0x006c },{ 0x006d,0x006d },{ 0x006e,0x006e },{ 0x006f,0x006f },
    { 0x0070,0x0070 },{ 0x0071,0x0071 },{ 0x0072,0x0072 },{ 0x0073,0x0073 },
    { 0x0074,0x0074 },{ 0x0075,0x0075 },{ 0x0076,0x0076 },{ 0x0077,0x0077 },
    { 0x0078,0x0078 },{ 0x0079,0x0079 },{ 0x007a,0x007a },{ 0x007b,0x007b },
    { 0x007c,0x007c },{ 0x007d,0x007d },{ 0x007e,0x007e },{ 0x007f,0x007f },
    { 0x0080,0x0080 },{ 0x0081,0x0081 },{ 0x0082,0x0082 },{ 0x0083,0x0083 },
    { 0x0084,0x0084 },{ 0x0085,0x0085 },{ 0x0086,0x0086 },{ 0x0087,0x0087 },
    { 0x0088,0x0088 },{ 0x0089,0x0089 },{ 0x008a,0x008a },{ 0x008b,0x008b },
    { 0x008c,0x008c },{ 0x008d,0x008d },{ 0x008e,0x008e },{ 0x008f,0x008f },
    { 0x0090,0x0090 },{ 0x0091,0x0091 },{ 0x0092,0x0092 },{ 0x0093,0x0093 },
    { 0x0094,0x0094 },{ 0x0095,0x0095 },{ 0x0096,0x0096 },{ 0x0097,0x0097 },
    { 0x0098,0x0098 },{ 0x0099,0x0099 },{ 0x009a,0x009a },{ 0x009b,0x009b },
    { 0x009c,0x009c },{ 0x009d,0x009d },{ 0x009e,0x009e },{ 0x009f,0x009f },
    { 0x00a0,0x00a0 },{ 0x00a3,0x00a3 },{ 0x00a4,0x00a4 },{ 0x00a7,0x00a7 },
    { 0x00a8,0x00a8 },{ 0x00ad,0x00ad },{ 0x00b0,0x00b0 },{ 0x00b2,0x00b2 },
    { 0x00b3,0x00b3 },{ 0x00b4,0x00b4 },{ 0x00b5,0x00b5 },{ 0x00b7,0x00b7 },
    { 0x00b8,0x00b8 },{ 0x00bd,0x00bd },{ 0x00c0,0x00c0 },{ 0x00c1,0x00c1 },
    { 0x00c2,0x00c2 },{ 0x00c4,0x00c4 },{ 0x00c7,0x00c7 },{ 0x00c8,0x00c8 },
    { 0x00c9,0x00c9 },{ 0x00ca,0x00ca },{ 0x00cb,0x00cb },{ 0x00cc,0x00cc },
    { 0x00cd,0x00cd },{ 0x00ce,0x00ce },{ 0x00cf,0x00cf },{ 0x00d1,0x00d1 },
    { 0x00d2,0x00d2 },{ 0x00d3,0x00d3 },{ 0x00d4,0x00d4 },{ 0x00d6,0x00d6 },
    { 0x00d7,0x00d7 },{ 0x00d9,0x00d9 },{ 0x00da,0x00da },{ 0x00db,0x00db },
    { 0x00dc,0x00dc },{ 0x00df,0x00df },{ 0x00e0,0x00e0 },{ 0x00e1,0x00e1 },
    { 0x00e2,0x00e2 },{ 0x00e4,0x00e4 },{ 0x00e7,0x00e7 },{ 0x00e8,0x00e8 },
    { 0x00e9,0x00e9 },{ 0x00ea,0x00ea },{ 0x00eb,0x00eb },{ 0x00ec,0x00ec },
    { 0x00ed,0x00ed },{ 0x00ee,0x00ee },{ 0x00ef,0x00ef },{ 0x00f1,0x00f1 },
    { 0x00f2,0x00f2 },{ 0x00f3,0x00f3 },{ 0x00f4,0x00f4 },{ 0x00f6,0x00f6 },
    { 0x00f7,0x00f7 },{ 0x00f9,0x00f9 },{ 0x00fa,0x00fa },{ 0x00fb,0x00fb },
    { 0x00fc,0x00fc },{ 0x0108,0x00c6 },{ 0x0109,0x00e6 },{ 0x010a,0x00c5 },
    { 0x010b,0x00e5 },{ 0x011c,0x00d8 },{ 0x011d,0x00f8 },{ 0x011e,0x00ab },
    { 0x011f,0x00bb },{ 0x0120,0x00d5 },{ 0x0121,0x00f5 },{ 0x0124,0x00a6 },
    { 0x0125,0x00b6 },{ 0x0126,0x00a1 },{ 0x0127,0x00b1 },{ 0x0130,0x00a9 },
    { 0x0131,0x00b9 },{ 0x0134,0x00ac },{ 0x0135,0x00bc },{ 0x015c,0x00de },
    { 0x015d,0x00fe },{ 0x015e,0x00aa },{ 0x015f,0x00ba },{ 0x016c,0x00dd },
    { 0x016d,0x00fd },{ 0x017b,0x00af },{ 0x017c,0x00bf },{ 0x02d8,0x00a2 },
    { 0x02d9,0x00ff },{ 0xffff,0x00d0 },{ 0xffff,0x00a5 },{ 0xffff,0x00be },
    { 0xffff,0x00ae },{ 0xffff,0x00e3 },{ 0xffff,0x00f0 },{ 0xffff,0x00c3 }
  };


/* ISO 8859-4:1988 character set. */

static const SshUInt16 ssh_charset_iso_8859_4[] =
  {
    0x0000,0x0001,0x0002,0x0003,0x0004,0x0005,0x0006,0x0007,
    0x0008,0x0009,0x000a,0x000b,0x000c,0x000d,0x000e,0x000f,
    0x0010,0x0011,0x0012,0x0013,0x0014,0x0015,0x0016,0x0017,
    0x0018,0x0019,0x001a,0x001b,0x001c,0x001d,0x001e,0x001f,
    0x0020,0x0021,0x0022,0x0023,0x0024,0x0025,0x0026,0x0027,
    0x0028,0x0029,0x002a,0x002b,0x002c,0x002d,0x002e,0x002f,
    0x0030,0x0031,0x0032,0x0033,0x0034,0x0035,0x0036,0x0037,
    0x0038,0x0039,0x003a,0x003b,0x003c,0x003d,0x003e,0x003f,
    0x0040,0x0041,0x0042,0x0043,0x0044,0x0045,0x0046,0x0047,
    0x0048,0x0049,0x004a,0x004b,0x004c,0x004d,0x004e,0x004f,
    0x0050,0x0051,0x0052,0x0053,0x0054,0x0055,0x0056,0x0057,
    0x0058,0x0059,0x005a,0x005b,0x005c,0x005d,0x005e,0x005f,
    0x0060,0x0061,0x0062,0x0063,0x0064,0x0065,0x0066,0x0067,
    0x0068,0x0069,0x006a,0x006b,0x006c,0x006d,0x006e,0x006f,
    0x0070,0x0071,0x0072,0x0073,0x0074,0x0075,0x0076,0x0077,
    0x0078,0x0079,0x007a,0x007b,0x007c,0x007d,0x007e,0x007f,
    0x0080,0x0081,0x0082,0x0083,0x0084,0x0085,0x0086,0x0087,
    0x0088,0x0089,0x008a,0x008b,0x008c,0x008d,0x008e,0x008f,
    0x0090,0x0091,0x0092,0x0093,0x0094,0x0095,0x0096,0x0097,
    0x0098,0x0099,0x009a,0x009b,0x009c,0x009d,0x009e,0x009f,
    0x00a0,0x0104,0x0138,0x0156,0x00a4,0x0128,0x013b,0x00a7,
    0x00a8,0x0160,0x0112,0x0122,0x0166,0x00ad,0x017d,0x203e,
    0x00b0,0x0105,0x02db,0x0157,0x00b4,0x0129,0x013c,0x02c7,
    0x00b8,0x0161,0x0113,0x0123,0x0167,0x014a,0x017e,0x014b,
    0x0100,0x00c1,0x00c2,0x00c3,0x00c4,0x00c5,0x00c6,0x012e,
    0x010c,0x00c9,0x0118,0x00cb,0x0116,0x00cd,0x00ce,0x012a,
    0x0110,0x0145,0x014c,0x0136,0x00d4,0x00d5,0x00d6,0x00d7,
    0x00d8,0x0172,0x00da,0x00db,0x00dc,0x0168,0x016a,0x00df,
    0x0101,0x00e1,0x00e2,0x00e3,0x00e4,0x00e5,0x00e6,0x012f,
    0x010d,0x00e9,0x0119,0x00eb,0x0117,0x00ed,0x00ee,0x012b,
    0x0111,0x0146,0x014d,0x0137,0x00f4,0x00f5,0x00f6,0x00f7,
    0x00f8,0x0173,0x00fa,0x00fb,0x00fc,0x0169,0x016b,0x02d9
  };

static const SshCharsetMap ssh_charset_iso_8859_4_map[] =
  {
    { 0x0000,0x0000 },{ 0x0001,0x0001 },{ 0x0002,0x0002 },{ 0x0003,0x0003 },
    { 0x0004,0x0004 },{ 0x0005,0x0005 },{ 0x0006,0x0006 },{ 0x0007,0x0007 },
    { 0x0008,0x0008 },{ 0x0009,0x0009 },{ 0x000a,0x000a },{ 0x000b,0x000b },
    { 0x000c,0x000c },{ 0x000d,0x000d },{ 0x000e,0x000e },{ 0x000f,0x000f },
    { 0x0010,0x0010 },{ 0x0011,0x0011 },{ 0x0012,0x0012 },{ 0x0013,0x0013 },
    { 0x0014,0x0014 },{ 0x0015,0x0015 },{ 0x0016,0x0016 },{ 0x0017,0x0017 },
    { 0x0018,0x0018 },{ 0x0019,0x0019 },{ 0x001a,0x001a },{ 0x001b,0x001b },
    { 0x001c,0x001c },{ 0x001d,0x001d },{ 0x001e,0x001e },{ 0x001f,0x001f },
    { 0x0020,0x0020 },{ 0x0021,0x0021 },{ 0x0022,0x0022 },{ 0x0023,0x0023 },
    { 0x0024,0x0024 },{ 0x0025,0x0025 },{ 0x0026,0x0026 },{ 0x0027,0x0027 },
    { 0x0028,0x0028 },{ 0x0029,0x0029 },{ 0x002a,0x002a },{ 0x002b,0x002b },
    { 0x002c,0x002c },{ 0x002d,0x002d },{ 0x002e,0x002e },{ 0x002f,0x002f },
    { 0x0030,0x0030 },{ 0x0031,0x0031 },{ 0x0032,0x0032 },{ 0x0033,0x0033 },
    { 0x0034,0x0034 },{ 0x0035,0x0035 },{ 0x0036,0x0036 },{ 0x0037,0x0037 },
    { 0x0038,0x0038 },{ 0x0039,0x0039 },{ 0x003a,0x003a },{ 0x003b,0x003b },
    { 0x003c,0x003c },{ 0x003d,0x003d },{ 0x003e,0x003e },{ 0x003f,0x003f },
    { 0x0040,0x0040 },{ 0x0041,0x0041 },{ 0x0042,0x0042 },{ 0x0043,0x0043 },
    { 0x0044,0x0044 },{ 0x0045,0x0045 },{ 0x0046,0x0046 },{ 0x0047,0x0047 },
    { 0x0048,0x0048 },{ 0x0049,0x0049 },{ 0x004a,0x004a },{ 0x004b,0x004b },
    { 0x004c,0x004c },{ 0x004d,0x004d },{ 0x004e,0x004e },{ 0x004f,0x004f },
    { 0x0050,0x0050 },{ 0x0051,0x0051 },{ 0x0052,0x0052 },{ 0x0053,0x0053 },
    { 0x0054,0x0054 },{ 0x0055,0x0055 },{ 0x0056,0x0056 },{ 0x0057,0x0057 },
    { 0x0058,0x0058 },{ 0x0059,0x0059 },{ 0x005a,0x005a },{ 0x005b,0x005b },
    { 0x005c,0x005c },{ 0x005d,0x005d },{ 0x005e,0x005e },{ 0x005f,0x005f },
    { 0x0060,0x0060 },{ 0x0061,0x0061 },{ 0x0062,0x0062 },{ 0x0063,0x0063 },
    { 0x0064,0x0064 },{ 0x0065,0x0065 },{ 0x0066,0x0066 },{ 0x0067,0x0067 },
    { 0x0068,0x0068 },{ 0x0069,0x0069 },{ 0x006a,0x006a },{ 0x006b,0x006b },
    { 0x006c,0x006c },{ 0x006d,0x006d },{ 0x006e,0x006e },{ 0x006f,0x006f },
    { 0x0070,0x0070 },{ 0x0071,0x0071 },{ 0x0072,0x0072 },{ 0x0073,0x0073 },
    { 0x0074,0x0074 },{ 0x0075,0x0075 },{ 0x0076,0x0076 },{ 0x0077,0x0077 },
    { 0x0078,0x0078 },{ 0x0079,0x0079 },{ 0x007a,0x007a },{ 0x007b,0x007b },
    { 0x007c,0x007c },{ 0x007d,0x007d },{ 0x007e,0x007e },{ 0x007f,0x007f },
    { 0x0080,0x0080 },{ 0x0081,0x0081 },{ 0x0082,0x0082 },{ 0x0083,0x0083 },
    { 0x0084,0x0084 },{ 0x0085,0x0085 },{ 0x0086,0x0086 },{ 0x0087,0x0087 },
    { 0x0088,0x0088 },{ 0x0089,0x0089 },{ 0x008a,0x008a },{ 0x008b,0x008b },
    { 0x008c,0x008c },{ 0x008d,0x008d },{ 0x008e,0x008e },{ 0x008f,0x008f },
    { 0x0090,0x0090 },{ 0x0091,0x0091 },{ 0x0092,0x0092 },{ 0x0093,0x0093 },
    { 0x0094,0x0094 },{ 0x0095,0x0095 },{ 0x0096,0x0096 },{ 0x0097,0x0097 },
    { 0x0098,0x0098 },{ 0x0099,0x0099 },{ 0x009a,0x009a },{ 0x009b,0x009b },
    { 0x009c,0x009c },{ 0x009d,0x009d },{ 0x009e,0x009e },{ 0x009f,0x009f },
    { 0x00a0,0x00a0 },{ 0x00a4,0x00a4 },{ 0x00a7,0x00a7 },{ 0x00a8,0x00a8 },
    { 0x00ad,0x00ad },{ 0x00b0,0x00b0 },{ 0x00b4,0x00b4 },{ 0x00b8,0x00b8 },
    { 0x00c1,0x00c1 },{ 0x00c2,0x00c2 },{ 0x00c3,0x00c3 },{ 0x00c4,0x00c4 },
    { 0x00c5,0x00c5 },{ 0x00c6,0x00c6 },{ 0x00c9,0x00c9 },{ 0x00cb,0x00cb },
    { 0x00cd,0x00cd },{ 0x00ce,0x00ce },{ 0x00d4,0x00d4 },{ 0x00d5,0x00d5 },
    { 0x00d6,0x00d6 },{ 0x00d7,0x00d7 },{ 0x00d8,0x00d8 },{ 0x00da,0x00da },
    { 0x00db,0x00db },{ 0x00dc,0x00dc },{ 0x00df,0x00df },{ 0x00e1,0x00e1 },
    { 0x00e2,0x00e2 },{ 0x00e3,0x00e3 },{ 0x00e4,0x00e4 },{ 0x00e5,0x00e5 },
    { 0x00e6,0x00e6 },{ 0x00e9,0x00e9 },{ 0x00eb,0x00eb },{ 0x00ed,0x00ed },
    { 0x00ee,0x00ee },{ 0x00f4,0x00f4 },{ 0x00f5,0x00f5 },{ 0x00f6,0x00f6 },
    { 0x00f7,0x00f7 },{ 0x00f8,0x00f8 },{ 0x00fa,0x00fa },{ 0x00fb,0x00fb },
    { 0x00fc,0x00fc },{ 0x0100,0x00c0 },{ 0x0101,0x00e0 },{ 0x0104,0x00a1 },
    { 0x0105,0x00b1 },{ 0x010c,0x00c8 },{ 0x010d,0x00e8 },{ 0x0110,0x00d0 },
    { 0x0111,0x00f0 },{ 0x0112,0x00aa },{ 0x0113,0x00ba },{ 0x0116,0x00cc },
    { 0x0117,0x00ec },{ 0x0118,0x00ca },{ 0x0119,0x00ea },{ 0x0122,0x00ab },
    { 0x0123,0x00bb },{ 0x0128,0x00a5 },{ 0x0129,0x00b5 },{ 0x012a,0x00cf },
    { 0x012b,0x00ef },{ 0x012e,0x00c7 },{ 0x012f,0x00e7 },{ 0x0136,0x00d3 },
    { 0x0137,0x00f3 },{ 0x0138,0x00a2 },{ 0x013b,0x00a6 },{ 0x013c,0x00b6 },
    { 0x0145,0x00d1 },{ 0x0146,0x00f1 },{ 0x014a,0x00bd },{ 0x014b,0x00bf },
    { 0x014c,0x00d2 },{ 0x014d,0x00f2 },{ 0x0156,0x00a3 },{ 0x0157,0x00b3 },
    { 0x0160,0x00a9 },{ 0x0161,0x00b9 },{ 0x0166,0x00ac },{ 0x0167,0x00bc },
    { 0x0168,0x00dd },{ 0x0169,0x00fd },{ 0x016a,0x00de },{ 0x016b,0x00fe },
    { 0x0172,0x00d9 },{ 0x0173,0x00f9 },{ 0x017d,0x00ae },{ 0x017e,0x00be },
    { 0x02c7,0x00b7 },{ 0x02d9,0x00ff },{ 0x02db,0x00b2 },{ 0x203e,0x00af }
  };


/* This functions performs binary search in the mapping tables of the
   character sets. The mapping tables are assumed to have size 2^n for
   some n > 2 */
static SshUInt32
ssh_charset_convert(const SshCharsetMap *table, size_t size, SshUInt32 unicode)
{
  SshUInt16 x, y, d;
  SshUInt32 ret;

  ret = 0xffffffff;
  y = table[0].unicode;
  if (y == unicode)
    ret = table[0].mapped & 0xffff;
  else
    {
      x = size >> 1;
      d = size >> 2;
      while (d)
        {
          y = table[x].unicode;
          if (y == unicode)
            break;
          if (y > unicode)
            x -= d;
          else
            x += d;
          d >>= 1;
        }
      y = table[x].unicode;
      if (y == unicode)
        ret = table[x].mapped & 0xffff;
    }
  return ret;
}

#define DEFINE_CONVERTER(source) \
static SshUInt32 ssh_charset_unicode_to_##source(SshUInt32 character) { \
  return ssh_charset_convert(ssh_charset_##source##_map, \
                      sizeof(ssh_charset_##source##_map) /  \
                      sizeof(ssh_charset_##source##_map[0]), \
                      character); \
}

DEFINE_CONVERTER(printable_string)
DEFINE_CONVERTER(visible)
DEFINE_CONVERTER(t61)

SshUInt32 ssh_charset_unicode_to_us_ascii(SshUInt32 unicode)
{
  if (unicode > 0x7f)
    return 0xffffffff;
  return unicode;
}

SshUInt32 ssh_charset_unicode_to_iso_8859_1(SshUInt32 unicode)
{
  if (unicode > 0xff)
    return 0xffffffff;
  return unicode;
}

SshUInt32 ssh_charset_unicode_to_iso_8859_15(SshUInt32 unicode)
{
  if (unicode < 0XA4)
    return unicode;

  switch (unicode)
    {
    case 0X20AC: return 0XA4;
    case 0X0160: return 0XA6;
    case 0X0161: return 0XA8;
    case 0X017D: return 0XB4;
    case 0X017E: return 0XB8;
    case 0X0152: return 0XBC;
    case 0X0153: return 0XBD;
    case 0X0178: return 0XBE;
    case  0XA4: case  0XA6: case  0XA8: case  0XB4:
    case  0XB8: case  0XBC: case  0XBD: case  0XBE:
      return 0xffffffff;
    }
  if (unicode > 0xff)
    return 0xffffffff;
  return unicode;
}

DEFINE_CONVERTER(iso_8859_2)
DEFINE_CONVERTER(iso_8859_3)
DEFINE_CONVERTER(iso_8859_4)

SshUInt32 ssh_charset_unicode_to_bmp(SshUInt32 unicode)
{
  if (unicode > 0xffff)
    return 0xffffffff;
  return unicode;
}

SshUInt32 ssh_charset_unicode_to_universal(SshUInt32 unicode)
{
  return unicode;
}



/***** Character set routines. */
/* Table to indentify printable chars. */
static const unsigned char ssh_charset_printable_chars[128] =
  {
    0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
    3,0,0,0,0,0,0,2,2,2,0,2,2,2,2,2,3,3,3,3,3,3,3,3,3,3,2,0,0,2,0,2,
    0,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,0,0,0,0,0,
    0,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,2,0,0,0,0,0
  };

Boolean ssh_str_isnumerical(unsigned char byte)
{
  if (byte > 127)
    return FALSE;
  if (ssh_charset_printable_chars[byte] & 1)
    return TRUE;
  return FALSE;
}

/* Functions to handle the printable chars table. */
Boolean ssh_str_isprintable(unsigned int byte)
{
  if (byte > 127)
    return FALSE;
  if (ssh_charset_printable_chars[byte] & 2)
    return TRUE;
  return FALSE;
}

SshCharset ssh_str_charset_get(SshStr str)
{
  if (str == NULL)
    return SSH_CHARSET_ANY;
  return str->charset;
}

/* Convert given string `str' into the character set `charset'. */
SshStr ssh_str_charset_convert(SshStr str, SshCharset charset)
{
  SshStr    new_str;
  SshUInt32 (*converter)(SshUInt32 unicode);
  SshUInt32 pos, value, letter;
  size_t i;

  if (str == NULL)
    return NULL;

  /* Set up the converter. */
  converter = NULL_FNPTR;

  switch (charset)
    {
    case SSH_CHARSET_PRINTABLE:
      converter = ssh_charset_unicode_to_printable_string;
      break;
    case SSH_CHARSET_VISIBLE:
      converter = ssh_charset_unicode_to_visible;
      break;
    case SSH_CHARSET_US_ASCII:
      converter = ssh_charset_unicode_to_us_ascii;
      break;
    case SSH_CHARSET_ISO_8859_1:
      converter = ssh_charset_unicode_to_iso_8859_1;
      break;
    case SSH_CHARSET_ISO_8859_2:
      converter = ssh_charset_unicode_to_iso_8859_2;
      break;
    case SSH_CHARSET_ISO_8859_3:
      converter = ssh_charset_unicode_to_iso_8859_3;
      break;
    case SSH_CHARSET_ISO_8859_4:
      converter = ssh_charset_unicode_to_iso_8859_4;
      break;
    case SSH_CHARSET_ISO_8859_15:
      converter = ssh_charset_unicode_to_iso_8859_15;
      break;
    case SSH_CHARSET_T61:
      converter = ssh_charset_unicode_to_t61;
      break;
    case SSH_CHARSET_BMP:
      converter = ssh_charset_unicode_to_bmp;
      break;
    case SSH_CHARSET_UNIVERSAL:
      converter = ssh_charset_unicode_to_universal;
      break;
    case SSH_CHARSET_UTF8:
      converter = ssh_charset_unicode_to_universal;
      break;
    default:
      ssh_fatal("ssh_str_charset_test: output charset unknown.");
    }

  /* Allocate a new string with suitable initial length. */
  if ((new_str = ssh_str_allocate(charset, str->str_length)) == NULL)
    return NULL;

  /* Run the tests. */
  switch (str->charset)
    {
    case SSH_CHARSET_PRINTABLE:
      for (i = 0; i < str->str_length; i++)
        {
          pos = str->ob.str_8bit[i];

          if (ssh_str_isprintable(pos) != TRUE)
            goto failed;

          value = ssh_charset_visible[pos];

          letter = (*converter)(value);
          if (letter == 0xffffffff)
            goto failed;

          if (!ssh_str_append_letter(new_str, letter))
            goto failed;
        }
      break;

    case SSH_CHARSET_VISIBLE:
      for (i = 0; i < str->str_length; i++)
        {
          pos = str->ob.str_8bit[i];

          if (pos > 0x7f)
            goto failed;

          value = ssh_charset_visible[pos];
          if (value == 0xffff)
            goto failed;

          letter = (*converter)(value);
          if (letter == 0xffffffff)
            goto failed;

          if (!ssh_str_append_letter(new_str, letter))
            goto failed;
        }
      break;

    case SSH_CHARSET_US_ASCII:
      for (i = 0; i < str->str_length; i++)
        {
          pos = str->ob.str_8bit[i];

          if (pos > 0x7f)
            goto failed;

          letter = (*converter)(pos);
          if (letter == 0xffffffff)
            goto failed;

          if (!ssh_str_append_letter(new_str, letter))
            goto failed;
        }
      break;

    case SSH_CHARSET_ISO_8859_1:
      for (i = 0; i < str->str_length; i++)
        {
          pos = str->ob.str_8bit[i];

          if (pos > 0xff)
            goto failed;

          letter = (*converter)(pos);
          if (letter == 0xffffffff)
            goto failed;

          if (!ssh_str_append_letter(new_str, letter))
            goto failed;
        }
      break;

    case SSH_CHARSET_ISO_8859_15:
      for (i = 0; i < str->str_length; i++)
        {
          pos = str->ob.str_8bit[i];

          if (pos > 0xff)
            goto failed;

          switch (pos)
            {
            case 0XA4: value = 0X20AC; break;
            case 0XA6: value = 0X0160; break;
            case 0XA8: value = 0X0161; break;
            case 0XB4: value = 0X017D; break;
            case 0XB8: value = 0X017E; break;
            case 0XBC: value = 0X0152; break;
            case 0XBD: value = 0X0153; break;
            case 0XBE: value = 0X0178; break;
            default: value = pos;
            }
          letter = (*converter)(value);
          if (letter == 0xffffffff)
            goto failed;

          if (!ssh_str_append_letter(new_str, letter))
            goto failed;
        }
      break;

    case SSH_CHARSET_ISO_8859_2:
      for (i = 0; i < str->str_length; i++)
        {
          pos = str->ob.str_8bit[i];
          value = ssh_charset_iso_8859_2[pos];
          if (value == 0xffff)
            goto failed;

          letter = (*converter)(value);
          if (letter == 0xffffffff)
            goto failed;

          if (!ssh_str_append_letter(new_str, letter))
            goto failed;
        }
      break;
    case SSH_CHARSET_ISO_8859_3:
      for (i = 0; i < str->str_length; i++)
        {
          pos = str->ob.str_8bit[i];
          value = ssh_charset_iso_8859_3[pos];
          if (value == 0xffff)
            goto failed;

          letter = (*converter)(value);
          if (letter == 0xffffffff)
            goto failed;

          if (!ssh_str_append_letter(new_str, letter))
            goto failed;
        }
      break;
    case SSH_CHARSET_ISO_8859_4:
      for (i = 0; i < str->str_length; i++)
        {
          pos = str->ob.str_8bit[i];
          value = ssh_charset_iso_8859_4[pos];
          if (value == 0xffff)
            goto failed;

          letter = (*converter)(value);
          if (letter == 0xffffffff)
            goto failed;

          if (!ssh_str_append_letter(new_str, letter))
            goto failed;
        }
      break;

    case SSH_CHARSET_T61:
      for (i = 0; i < str->str_length; i++)
        {
          pos = str->ob.str_8bit[i];
          value = ssh_charset_convert(ssh_charset_t61_unicode_map,
                                      sizeof(ssh_charset_t61_unicode_map) /
                                      sizeof(ssh_charset_t61_unicode_map[0]),
                                      pos);
          if ((value & 0xff00) == 0xe000 && (i+1) < str->str_length)
            {
              /* extented. use two bytes as pos now. */
              pos = (pos << 8) | str->ob.str_8bit[i+1];
              value =
                ssh_charset_convert(ssh_charset_t61_unicode_map,
                                    sizeof(ssh_charset_t61_unicode_map) /
                                    sizeof(ssh_charset_t61_unicode_map[0]),
                                    pos);
              i++;
            }

          if (value == 0xffff)
            goto failed;

          letter = (*converter)(value);
          if (letter == 0xffffffff)
            goto failed;

          if (!ssh_str_append_letter(new_str, letter))
            goto failed;
        }
      break;

    case SSH_CHARSET_BMP:
      for (i = 0; i < str->str_length; i++)
        {
          value = str->ob.str_16bit[i];

          letter = (*converter)(value);
          if (letter == 0xffffffff)
            goto failed;

          if (!ssh_str_append_letter(new_str, letter))
            goto failed;
        }
      break;

    case SSH_CHARSET_UNIVERSAL:
    case SSH_CHARSET_UTF8:
      for (i = 0; i < str->str_length; i++)
        {
          value = str->ob.str_32bit[i];
          if (value > 0xffff)
            goto failed;

          letter = (*converter)(value);
          if (letter == 0xffffffff)
            goto failed;

          if (!ssh_str_append_letter(new_str, letter))
            goto failed;
        }
      break;

    default:
      ssh_fatal("ssh_str_charset_test: character set %u not defined.\n",
                str->charset);
      break;
    }

  return new_str;

 failed:
  ssh_str_free(new_str);
  return NULL;
}

/***** Routines for encoding and decoding. */

/* Note: this idea of converting everything to some internal
   representation takes CPU and more memory. There are other ways
   which could be used such as lazily wait until the data is needed
   and convert only then.  However, I feel that at the moment this is
   easier to manage internally and thus I will get this code quicker
   ready. Later if you or I have time this can be changed.

   The UTF-8 coding has the advantage of taking less space than usual
   for universal strings, however, here as you can see I convert them
   to universal strings and thus take potentially even 4 times as much
   space. Too bad, but it's still easier... */


/* Handle the 16-bit strings. */

SshStr
ssh_str_decode_bmp(unsigned char *str, size_t str_length)
{
  SshStr bmp_str;
  size_t i;

  /* Check for error. */
  if (!str || str_length & 0x1 || str_length == 0)
    return NULL;

  /* Allocate the BMP string. */
  if ((bmp_str = ssh_str_allocate(SSH_CHARSET_BMP, str_length/2)) == NULL)
    return NULL;

  /* Decode the byte string. Value MSB first. */
  for (i = 0; i < str_length; i += 2)
    if (!ssh_str_append_letter(bmp_str,
                               (((SshUInt16)str[i    ]) << 8) |
                               (((SshUInt16)str[i + 1])     )))
      {
        ssh_str_free(bmp_str);
        return NULL;
      }
  return bmp_str;
}

unsigned char *
ssh_str_encode_bmp(SshStr bmp_str, size_t *str_length)
{
  unsigned char *str;
  SshUInt32 letter;
  size_t i;

  if (bmp_str == NULL || bmp_str->str_length == 0)
    {
      *str_length = 0;
      return NULL;
    }

  /* Allocate the string and convert. */
  if ((str = ssh_malloc(bmp_str->str_length * 2)) != NULL)
    {
      for (i = 0; ssh_str_get_letter(bmp_str, i, &letter); i++)
        {
          str[(i<<1) + 0] = (letter >> 8) & 0xff;
          str[(i<<1) + 1] = (letter     ) & 0xff;
        }
      *str_length = bmp_str->str_length * 2;
    }
  else
    *str_length = 0;
  return str;
}

/* Handle the 32-bit strings. */
SshStr
ssh_str_decode_universal(unsigned char *str, size_t str_length)
{
  SshStr universal_str;
  size_t i;

  /* Check for error. */
  if (str_length & 0x3 || str_length == 0)
    return NULL;

  /* Allocate the UNIVERSAL string and decode to MSB first. */
  if ((universal_str = ssh_str_allocate(SSH_CHARSET_UNIVERSAL, str_length/4))
      == NULL)
    return NULL;
  for (i = 0; i < str_length; i += 4)
    if (!ssh_str_append_letter(universal_str,
                               (((SshUInt32)str[i    ]) << 24) |
                               (((SshUInt32)str[i + 1]) << 16) |
                               (((SshUInt32)str[i + 2]) <<  8) |
                               (((SshUInt32)str[i + 3])      )))
      {
        ssh_str_free(universal_str);
        return NULL;
      }
  return universal_str;
}

unsigned char *
ssh_str_encode_universal(SshStr universal_str, size_t *str_length)
{
  unsigned char *str;
  SshUInt32 letter;
  size_t i;

  if (universal_str == NULL || universal_str->str_length == 0)
    {
      *str_length = 0;
      return NULL;
    }

  /* Allocate the string and convert. */
  if ((str = ssh_malloc(universal_str->str_length * 4)) != NULL)
    {
      for (i = 0; ssh_str_get_letter(universal_str, i, &letter); i++)
        {
          str[(i<<2) + 0] = (letter >> 24) & 0xff;
          str[(i<<2) + 1] = (letter >> 16) & 0xff;
          str[(i<<2) + 2] = (letter >>  8) & 0xff;
          str[(i<<2) + 3] = (letter      ) & 0xff;
        }
      *str_length = universal_str->str_length * 4;
    }
  else
    *str_length = 0;
  return str;
}

/* Handle the UTF-8 encoded strings. */
SshStr
ssh_str_decode_utf8(unsigned char *str, size_t str_length)
{
  SshStr utf8_str;
  SshUInt32 k, bits, byte, cur;
  size_t i, len, maxchar = 0;

  /* Compute the number of the characters in string. */
  for (i = 0, len = 0; i < str_length; i += k, len++)
    {
      byte = str[i];

      /* Compute the length of this character in bytes. */
      for (k = 0, bits = 0x80; byte & bits; bits >>= 1, k++)
        ;

      /* Check for length. */
      if (i + k > str_length)
        return NULL;

      if (k == 0)
        k++;

      if (k > maxchar)
        maxchar = k;
    }

  if (maxchar == 1)
    {
      /* UTF-8 string with single byte per charater is actually
         pure US ASCII and we can store it as such. */
      if ((utf8_str = ssh_str_allocate(SSH_CHARSET_US_ASCII, 0)) == NULL)
        return NULL;
      if ((utf8_str->ob.str_8bit = ssh_memdup(str, str_length)) == NULL)
        goto failed;
      utf8_str->str_length = str_length;
      utf8_str->mem_length = str_length;
      return utf8_str;
    }

  /* Initialize the UTF-8 string. */
  if ((utf8_str = ssh_str_allocate(SSH_CHARSET_UTF8, len)) == NULL)
    return NULL;

  for (i = 0; i < str_length; )
    {
      byte = str[i];

      /* Compute the length of this character in bytes. */
      for (k = 0, bits = 0x80; byte & bits; bits >>= 1, k++)
        ;

      /* Check for length. */
      if (i + k > str_length)
        goto failed;

      if (k == 0)
        k++;

      switch (k)
        {
        case 1:
          cur  = ((SshUInt32)str[i    ] & 0x7f);
          break;
        case 2:
          cur  = ((SshUInt32)str[i + 1] & 0x3f);
          cur |= ((SshUInt32)str[i    ] & 0x1f) << 6;
          break;
        case 3:
          cur  = ((SshUInt32)str[i + 2] & 0x3f);
          cur |= ((SshUInt32)str[i + 1] & 0x3f) << 6;
          cur |= ((SshUInt32)str[i    ] & 0x0f) << 12;
          break;
        case 4:
          cur  = ((SshUInt32)str[i + 3] & 0x3f);
          cur |= ((SshUInt32)str[i + 2] & 0x3f) << 6;
          cur |= ((SshUInt32)str[i + 1] & 0x3f) << 12;
          cur |= ((SshUInt32)str[i    ] & 0x07) << 18;
          break;
        case 5:
          cur  = ((SshUInt32)str[i + 4] & 0x3f);
          cur |= ((SshUInt32)str[i + 3] & 0x3f) << 6;
          cur |= ((SshUInt32)str[i + 2] & 0x3f) << 12;
          cur |= ((SshUInt32)str[i + 1] & 0x3f) << 18;
          cur |= ((SshUInt32)str[i    ] & 0x03) << 24;
          break;
        case 6:
          cur  = ((SshUInt32)str[i + 5] & 0x3f);
          cur |= ((SshUInt32)str[i + 4] & 0x3f) << 6;
          cur |= ((SshUInt32)str[i + 3] & 0x3f) << 12;
          cur |= ((SshUInt32)str[i + 2] & 0x3f) << 18;
          cur |= ((SshUInt32)str[i + 1] & 0x3f) << 24;
          cur |= ((SshUInt32)str[i + 0] & 0x01) << 31;
          break;
        default:
          goto failed;
          break;
        }
      if (!ssh_str_append_letter(utf8_str, cur))
        goto failed;
      i += k;
    }

  /* Return with the string. */
  return utf8_str;

 failed:
  ssh_str_free(utf8_str);
  return NULL;
}

unsigned char *
ssh_str_encode_utf8(SshStr utf8_str, size_t *str_length)
{
  unsigned char *str;
  SshStr output_str;
  SshUInt32 cur;
  size_t i;

  if (utf8_str == NULL)
    {
      *str_length = 0;
      return NULL;
    }

  /* Initialize the str buffer. We use the 8-bit ISO-8859-1 to act
     as an ordinary octet string. TODO. Add a binary string type. */
  if ((output_str = ssh_str_allocate(SSH_CHARSET_ISO_8859_1,
                                     utf8_str->str_length)) == NULL)
    {
      *str_length = 0;
      return NULL;
    }

  for (i = 0; ssh_str_get_letter(utf8_str, i, &cur); i++)
    {
      /* First the most frequent customer. */
      if (cur < 0x80)
        {
          if (!ssh_str_append_letter(output_str, (cur >>  0) & 0x7f))
            goto failed;
          continue;
        }

      /* One could count the highest leading zeroes of the 'cur' to
         quickly jump to the following cases. However, in C we'd need
         a byte table and we have tables enough in this file.

         If speed (or lack of it) becomes an issue then that could be
         done. */

      if (cur < 0x800)
        {
          if (ssh_str_append_letter(output_str, ((cur >>  6) & 0x1f) | 0xc0) &&
              ssh_str_append_letter(output_str, ((cur >>  0) & 0x3f) | 0x80))
            continue;
          else
            goto failed;
        }
      if (cur < 0x10000)
        {
          if (ssh_str_append_letter(output_str, ((cur >> 12) & 0x0f) | 0xe0) &&
              ssh_str_append_letter(output_str, ((cur >>  6) & 0x3f) | 0x80) &&
              ssh_str_append_letter(output_str, ((cur >>  0) & 0x3f) | 0x80))
            continue;
          else
            goto failed;
        }
      if (cur < 0x200000)
        {
          if (ssh_str_append_letter(output_str, ((cur >> 18) & 0x07) | 0xf0) &&
              ssh_str_append_letter(output_str, ((cur >> 12) & 0x3f) | 0x80) &&
              ssh_str_append_letter(output_str, ((cur >>  6) & 0x3f) | 0x80) &&
              ssh_str_append_letter(output_str, ((cur >>  0) & 0x3f) | 0x80))
            continue;
          else
            goto failed;
        }
      if (cur < 0x4000000)
        {
          if (ssh_str_append_letter(output_str, ((cur >> 24) & 0x03) | 0xf8) &&
              ssh_str_append_letter(output_str, ((cur >> 18) & 0x3f) | 0x80) &&
              ssh_str_append_letter(output_str, ((cur >> 12) & 0x3f) | 0x80) &&
              ssh_str_append_letter(output_str, ((cur >>  6) & 0x3f) | 0x80) &&
              ssh_str_append_letter(output_str, ((cur >>  0) & 0x3f) | 0x80))
            continue;
          else
            goto failed;
        }
      if (cur < 0x80000000)
        {
          if (ssh_str_append_letter(output_str, ((cur >> 30) & 0x01) | 0xfc) &&
              ssh_str_append_letter(output_str, ((cur >> 24) & 0x3f) | 0x80) &&
              ssh_str_append_letter(output_str, ((cur >> 18) & 0x3f) | 0x80) &&
              ssh_str_append_letter(output_str, ((cur >> 12) & 0x3f) | 0x80) &&
              ssh_str_append_letter(output_str, ((cur >>  6) & 0x3f) | 0x80) &&
              ssh_str_append_letter(output_str, ((cur >>  0) & 0x3f) | 0x80))
            continue;
          else
            goto failed;
        }

    failed:
      ssh_str_free(output_str);
      *str_length = 0;
      return NULL;
    }

  /* Get the string directly. */
  str = ssh_str_get_data(output_str, str_length);
  if (str)
    str[*str_length] = '\0';
  ssh_str_free_wrapper(output_str);
  return str;
}

/* Main string routines. */
int ssh_str_null(SshStr str)
{
  switch (str->bits)
    {
    case 8:
      return (str->ob.str_8bit == NULL) ? 1 : 0;
    case 16:
      return (str->ob.str_16bit == NULL) ? 1 : 0;
    case 32:
      return (str->ob.str_32bit == NULL) ? 1 : 0;
    default:
      break;
    }
  return 1;
}

/* Build a string from given data, this function doesn't make copies. */
SshStr ssh_str_make(SshCharset charset,
                    unsigned char *str, size_t str_length)
{
  SshStr new_str = NULL;

  if (str == NULL)
    return NULL;

  /* Handle the possible encoding, and convert to form which is easiest
     for us to handle. */

  switch (charset)
    {
    case SSH_CHARSET_PRINTABLE:
    case SSH_CHARSET_VISIBLE:
    case SSH_CHARSET_US_ASCII:
    case SSH_CHARSET_ISO_8859_1:
    case SSH_CHARSET_ISO_8859_2:
    case SSH_CHARSET_ISO_8859_3:
    case SSH_CHARSET_ISO_8859_4:
    case SSH_CHARSET_ISO_8859_15:
    case SSH_CHARSET_T61:
      /* No encoding, just a plain 8-bit string. */
      if ((new_str = ssh_str_allocate(charset, 0)) != NULL)
        {
          new_str->ob.str_8bit   = str;
          new_str->str_length = str_length;
          new_str->mem_length = str_length;
        }
      else
        {
          ssh_free(str);
        }
      break;
    case SSH_CHARSET_BMP:
      /* This has some encoding assuming msb first order. */
      new_str = ssh_str_decode_bmp(str, str_length);
      ssh_free(str);
      break;
    case SSH_CHARSET_UNIVERSAL:
      /* No special encoding. */
      new_str = ssh_str_decode_universal(str, str_length);
      ssh_free(str);
      break;
    case SSH_CHARSET_UTF8:
      /* Encoded, remove the encoding. Currently we assume that given
         UTF-8 string it is a ok to keep it stored as 32-bit universal
         string. */
      new_str = ssh_str_decode_utf8(str, str_length);
      ssh_free(str);
      break;
    default:
      return NULL;
    }

  if (new_str == NULL)
    return NULL;

  /* If no string available then return no string either. */
  if (ssh_str_null(new_str))
    {
      ssh_str_free(new_str);
      return NULL;
    }

  /* Return allocated string. */
  return new_str;
}

/* Copy given string. */
SshStr ssh_str_dup(SshStr str)
{
  SshStr newp;

  if (str == NULL)
    return NULL;

  newp = ssh_malloc(sizeof(*newp));
  if (!newp)
    return NULL;

  newp->charset = str->charset;
  newp->bits = str->bits;
  newp->str_length = str->str_length;

  switch (str->bits)
    {
    case 8:
      if ((newp->ob.str_8bit =
           ssh_memdup(str->ob.str_8bit,
                      str->str_length * sizeof(unsigned char))) == NULL)
        {
          ssh_free(newp);
          return NULL;
        }
      break;
    case 16:
      if ((newp->ob.str_16bit =
           ssh_memdup(str->ob.str_16bit,
                      str->str_length * sizeof(SshUInt16))) == NULL)
        {
          ssh_free(newp);
          return NULL;
        }
      break;
    case 32:
      if ((newp->ob.str_32bit =
           ssh_memdup(str->ob.str_32bit,
                      str->str_length * sizeof(SshUInt32))) == NULL)
        {
          ssh_free(newp);
          return NULL;
        }
      break;
    default:
      memset(&newp->ob, 0, sizeof(newp->ob));
      break;
    }

  return newp;
}

/* Returns the string length. This should be correct for each and every
   one of the strings. However, that is not the case because currently
   we handle the T.61 case imperfectly. */
size_t ssh_str_length(SshStr str)
{
  if (str == NULL)
    return 0;
  return str->str_length;
}

SshStrOrdRel ssh_str_cmp_internal(SshStr op1, SshStr op2)
{
  size_t i, len;

  if (op1->charset != op2->charset)
    ssh_fatal("ssh_str_cmp_internal:"
              " arguments not of the same character set.");

  /* Find the correct length. */
  len = (op1->str_length <= op2->str_length) ? op1->str_length :
    op2->str_length;

#define CMP(nbits)                                                      \
  for (i = 0; i < len; i++)                                             \
    {                                                                   \
      if (op1->ob.str_## nbits ##bit[i] != op2->ob.str_## nbits ##bit[i])    \
        {                                                               \
          if (op1->ob.str_## nbits ##bit[i] > op2->ob.str_## nbits ##bit[i]) \
            return SSH_STR_ORDREL_GT;                                   \
          return SSH_STR_ORDREL_LT;                                     \
        }                                                               \
    }

  switch (op1->bits)
    {
    case  8: CMP(8);  break;
    case 16: CMP(16); break;
    case 32: CMP(32); break;
    }

  if (op1->str_length > len)
    return SSH_STR_ORDREL_GT;
  if (op2->str_length > len)
    return SSH_STR_ORDREL_LT;

  return SSH_STR_ORDREL_EQ;
}

SshStrOrdRel ssh_str_cmp(SshStr op1, SshStr op2)
{
  SshStrOrdRel rv;

  /* We take the convention that NULL input is infinitesimally small. */
  if (op1 == NULL || op2 == NULL)
    {
      if (op1 != NULL)
        return SSH_STR_ORDREL_GT;
      if (op2 != NULL)
        return SSH_STR_ORDREL_LT;
      return SSH_STR_ORDREL_EQ;
    }

  /* First attempt to convert to common character set. */
  if (op1->charset != op2->charset)
    {
      SshStr newp;

      newp = ssh_str_charset_convert(op2, op1->charset);
      if (newp != NULL)
        {
          rv = ssh_str_cmp_internal(op1, newp);
          ssh_str_free(newp);
          goto finished;
        }
      newp = ssh_str_charset_convert(op1, op2->charset);
      if (newp != NULL)
        {
          rv = ssh_str_cmp_internal(newp, op2);
          ssh_str_free(newp);
          goto finished;
        }
      /* Incomparable. */
      return SSH_STR_ORDREL_IC;
    }

  rv = ssh_str_cmp_internal(op1, op2);

 finished:
  return rv;
}

/* Get an unsigned char string. This routine doesn't necessarily
   return canonicalized strings, but does in fact force the output to
   be of correct byte order. */
unsigned char *
ssh_str_get(SshStr str, size_t *new_str_length)
{
  unsigned char *new_str = NULL;

  if (str == NULL)
    {
      *new_str_length = 0;
      return NULL;
    }

  switch (str->bits)
    {
    case 8:
      if ((new_str = ssh_malloc(str->str_length + 1)) != NULL)
        {
          memcpy(new_str, str->ob.str_8bit, str->str_length);
          *new_str_length = str->str_length;
          new_str[str->str_length] = '\0';
        }
      else
        *new_str_length = 0;
      break;

    default:
      switch (str->charset)
        {
        case SSH_CHARSET_BMP:
          new_str = ssh_str_encode_bmp(str, new_str_length);
          break;

        case SSH_CHARSET_UNIVERSAL:
          new_str = ssh_str_encode_universal(str, new_str_length);
          break;

        case SSH_CHARSET_UTF8:
          new_str = ssh_str_encode_utf8(str, new_str_length);
          break;

        default:
          ssh_fatal("ssh_str_get: unknown character set %u (%u bit chars).",
                    str->charset, str->bits);
          break;
        }
      break;
    }
  return new_str;
}

/* Gets as canonical string out of the given string as possible. */
unsigned char *
ssh_str_get_canonical(SshStr str, size_t *str_length)
{
  SshStr temp;
  unsigned char *new_str;
  SshUInt32 cur, space, first;
  size_t i;

  if (str == NULL)
    {
      *str_length = 0;
      return NULL;
    }

  /* Allocate temporary string and initialize */
  if ((temp = ssh_str_allocate(str->charset, str->str_length)) == NULL)
    return NULL;

  first = 1;
  space = 0;

  /* Remark. This code needs to be rewritten in more modular fashion. */
  for (i = 0; i < str->str_length; i++)
    {
      if (!ssh_str_get_letter(str, i, &cur))
        break;

      switch (str->charset)
        {
        case SSH_CHARSET_PRINTABLE:
	case SSH_CHARSET_US_ASCII:
	  if (isupper(cur))
	    cur = tolower(cur);
          break;
        default:
          break;
        }

      switch (cur)
        {
        case 0x20:
          space = 1;
          break;
        default:
          if (space && !first)
            if (!ssh_str_append_letter(temp, 0x20))
              {
                ssh_str_free(temp);
                return NULL;
              }
          if (!ssh_str_append_letter(temp, cur))
            {
              ssh_str_free(temp);
              return NULL;
            }
          space = 0;
          first = 0;
          break;
        }
    }

  /* Convert to suitable output. */
  new_str = ssh_str_get(temp, str_length);
  ssh_str_free(temp);

  return new_str;
}

/* Test if charset can be used to represent given string. This
   function is only provided for library backwards compatibility and
   is not optimal. It will actually do the conversion so if you'd need
   the resulting string it is better to call ssh_str_charset_convert
   directly. */
Boolean ssh_str_charset_test(SshStr str, SshCharset charset)
{
  SshStr newstr = ssh_str_charset_convert(str, charset);
  Boolean rv;

  if (newstr)
    {
      ssh_str_free(newstr);
      rv = TRUE;
    }
  else
    rv = FALSE;

  return rv;
}

/* sshstr.c */
