/*
 *
 * dtd-compress.c
 *
 * Copyright:
 *       Copyright (c) 2002-2007 SFNT Finland Oy.
 *       All rights reserved.
 *
 * A simple compress / bytefy utility for XML DTDs.
 *
 */

#include "sshincludes.h"
#include "sshgetopt.h"

/************************** Types and definitions ***************************/

#define IS_SPACE(ch) \
((ch) == 0x20 || (ch) == 0x9 || (ch) == 0xd || (ch) == 0xa)

struct SshDtdCompressOutputRec
{
  void *(*start)(FILE *fp);
  void (*output)(FILE *fp, int ch, void *ctx);
  void (*end)(FILE *fp, void *ctx);
};

typedef struct SshDtdCompressOutputRec SshDtdCompressOutputStruct;
typedef struct SshDtdCompressOutputRec *SshDtdCompressOutput;


/***************************** Static variables *****************************/

char *program;

char *output_file = NULL;

char *array_name;

/* Output method. */
SshDtdCompressOutput output;


/******************************* Print output *******************************/

static void *
output_print_start(FILE *fp)
{
  return NULL;
}

static void
output_print_output(FILE *fp, int ch, void *ctx)
{
  fputc(ch, fp);
}

static void
output_print_end(FILE *fp, void *ctx)
{
}

static SshDtdCompressOutputStruct output_print =
{
  output_print_start,
  output_print_output,
  output_print_end,
};


/****************************** Bytefy output *******************************/

static int bytes = 0;

static void *
output_bytefy_start(FILE *fp)
{
  fprintf(fp, "/* This file is automatically generated.  Do not edit! */\n\n");
  fprintf(fp, "#include <sshincludes.h>\n\n");
  fprintf(fp, "const unsigned char %s[] =\n{", array_name);
  return NULL;
}

static void
output_bytefy_output(FILE *fp, int ch, void *ctx)
{
  if ((bytes % 10) == 0)
    fprintf(fp, "\n ");

  fprintf(fp, " 0x%02x,", ch);

  bytes++;
}

static void
output_bytefy_end(FILE *fp, void *ctx)
{
  fprintf(fp, "\n};\n\n");
  fprintf(fp, "const size_t %s_len = %u;\n", array_name, bytes);
}

static SshDtdCompressOutputStruct output_bytefy =
{
  output_bytefy_start,
  output_bytefy_output,
  output_bytefy_end,
};


/************************** Static help functions ***************************/

static void
usage(void)
{
  fprintf(stdout, "\
Usage: %s [OPTION]... [DTD]\n\
  -b ARRAY      bytefy input into a C array ARRAY\n\
  -h            print this help and exit\n\
  -o FILE       save output to file FILE\n\
",
          program);
}

static void output_ch(FILE *ofp, int ch, void *ctx)
{
  static int last_ch = '\n';

  if (last_ch == '\n' && ch == '\n')
    return;

  last_ch = ch;
  (*output->output)(ofp, ch, ctx);
}


static void
process_input(FILE *ifp, FILE *ofp, void *ctx)
{
  int ch;

  while ((ch = getc(ifp)) != EOF)
    {
      if (IS_SPACE(ch))
        {
          int newline = 0;

          do
            {
              if (ch == '\n')
                newline = 1;
            }
          while ((ch = getc(ifp)) != EOF && IS_SPACE(ch));

          if (ch != EOF)
            ungetc(ch, ifp);

          if (newline)
            ch = '\n';
          else
            ch = ' ';
        }
      else if (ch == '<')
        {
          ch = getc(ifp);
          if (ch == '!')
            {
              ch = getc(ifp);
              if (ch == '-')
                {
                  ch = getc(ifp);
                  if (ch == '-')
                    {
                      /* This is a comment. */
                      while ((ch = getc(ifp)) != EOF)
                        {
                          if (ch == '-')
                            {
                            hyphen_seen:
                              ch = getc(ifp);
                              if (ch == '-')
                                {
                                  ch = getc(ifp);
                                  if (ch == '>')
                                    /* End of comment found. */
                                    goto next_at_main_loop;
                                  else
                                    goto hyphen_seen;
                                }
                            }
                        }
                    }
                  else
                    {
                      output_ch(ofp, '<', ctx);
                      output_ch(ofp, '!', ctx);
                      output_ch(ofp, '-', ctx);
                    }
                }
              else
                {
                      output_ch(ofp, '<', ctx);
                      output_ch(ofp, '!', ctx);
                }
            }
          else
            {
              output_ch(ofp, '<', ctx);
            }
        }

      output_ch(ofp, ch, ctx);

    next_at_main_loop:
      ;
    }
}


/********************************* The main *********************************/

int
main(int argc, char *argv[])
{
  int opt;
  void *ctx;
  FILE *ifp;
  FILE *ofp;

  program = strrchr(argv[0], '/');
  if (program)
    program++;
  else
    program = argv[0];

  output = &output_print;

  /* Parse options. */
  while ((opt = ssh_getopt(argc, argv, "b:ho:", NULL)) != EOF)
    {
      switch (opt)
        {
        case 'b':
          output = &output_bytefy;
          array_name = ssh_optarg;
          break;

        case 'h':
          usage();
          exit(0);
          break;

        case 'o':
          output_file = ssh_optarg;
          break;

        case '?':
          fprintf(stderr, "Try `%s -h' for more information.\n", program);
          exit (1);
          break;
        }
    }

  if (output_file)
    {
      ofp = fopen(output_file, "wb");
      if (ofp == NULL)
        {
          fprintf(stderr, "%s: Could not create output file `%s': %s\n",
                  program, output_file, strerror(errno));
          exit(1);
        }
    }
  else
    ofp = stdout;

  /* Notify output module about start. */
  ctx = (*output->start)(ofp);

  /* Process all input files. */
  if (ssh_optind >= argc)
    {
      process_input(stdin, ofp, ctx);
    }
  else
    {
      for (; ssh_optind < argc; ssh_optind++)
        {
          ifp = fopen(argv[ssh_optind], "rb");
          if (ifp == NULL)
            {
              fprintf(stderr, "%s: Could not open input file `%s': %s\n",
                      program, argv[ssh_optind], strerror(errno));
              continue;
            }

          process_input(ifp, ofp, ctx);
          fclose(ifp);
        }
    }

  /* Processing done. */
  (*output->end)(ofp, ctx);

  if (output_file)
    fclose(ofp);

  return 0;
}
