/*
* Copyright (c) 2013 Qualcomm Atheros, Inc..
* All Rights Reserved.
* Qualcomm Atheros Confidential and Proprietary.
*/

#if UNIFIED_SMARTANTENNA

#include <ieee80211_var.h>
#include <ol_if_athvar.h>
#include <if_smart_ant.h>
#include <ieee80211_smart_ant_api.h>
#include <ol_txrx_types.h>
#include "ol_tx_desc.h"
#include "ol_if_smart_ant.h"

uint32_t g_tx_ppdu_end[MAX_TX_PPDU_SIZE];
wdi_event_subscribe SMART_ANT_TX_SUBSCRIBER;

static int
wmi_unified_smart_ant_assoc_handler(ol_scn_t scn, u_int8_t *data, u_int16_t datalen, void *context)
{
    struct ieee80211com *ic = &scn->sc_ic;
    A_UINT8 peer_macaddr[ATH_MAC_LEN];
    struct ieee80211_node *ni = NULL;
    struct sa_rate_cap rate_cap;
    int htindex = 0, i, j;
    uint8_t shift =0;
    wmi_peer_ratecode_list_event_t *rate_event = (wmi_peer_ratecode_list_event_t *)data;
    WMI_MAC_ADDR_TO_CHAR_ARRAY(&rate_event->peer_macaddr,peer_macaddr);
    
    ni = ieee80211_find_node(&ic->ic_sta, peer_macaddr);
    if (!ni) { 
        return -1; 
    } 

    /* In AP mode if security is enabled, do node connect indication when node is authorized
     * to send data.In STA mode WMI_PEER_RATECODE_LIST_EVENTID is received only once and
     * even before node gets authorized, so in STA mode do not check authorized flag.
     */
    if (((ni->ni_vap)->iv_opmode == IEEE80211_M_HOSTAP) && !ieee80211_node_is_authorized(ni)) {
        ieee80211_free_node(ni);
        return -1; 
    }

    htindex = 0;
    rate_cap.ratecount[0] = ((rate_event->peer_rate_info.ratecount) & MASK_BYTE);
    rate_cap.ratecount[1] = ((rate_event->peer_rate_info.ratecount >> 8) & MASK_BYTE);
    rate_cap.ratecount[2] = ((rate_event->peer_rate_info.ratecount >> 16) & MASK_BYTE);
    rate_cap.ratecount[3] = ((rate_event->peer_rate_info.ratecount >> 24) & MASK_BYTE);

    if (rate_cap.ratecount[0]) {
        for (i = 0; i < MAX_LEGACY_RATE_DWORDS; i++) {
            for (j = 0;j < BYTES_IN_DWORD; j++) {
                rate_cap.ratecode_legacy[htindex] = ((rate_event->peer_rate_info.ratecode_legacy[i] >> (8*j)) & MASK_BYTE);
                htindex++;
            }
        }
    }

    htindex = 0;
    for (i = 0; i < MAX_HT_RATE_DWORDS; i++) {
        for (j = 0; j < BYTES_IN_DWORD; j++) {
            shift = (8*j);
            rate_cap.ratecode_20[htindex] = ((rate_event->peer_rate_info.ratecode_20[i] >> (shift)) & MASK_BYTE);
            rate_cap.ratecode_40[htindex] = ((rate_event->peer_rate_info.ratecode_40[i] >> (shift)) & MASK_BYTE);
            rate_cap.ratecode_80[htindex] = ((rate_event->peer_rate_info.ratecode_80[i] >> (shift)) & MASK_BYTE);
            htindex++;
        }
    }

    /* node connect */
    ieee80211_smart_ant_node_connect(ni, &rate_cap);
    ieee80211_free_node(ni);
    return 0;
}

void ol_ath_smart_ant_enable(struct ieee80211com *ic, int enable, int mode, uint32_t rx_antenna)
{
    /* Send WMI COMMAND to Enable */
    struct ol_ath_softc_net80211 *scn = OL_ATH_SOFTC_NET80211(ic);
    wmi_pdev_smart_ant_enable_cmd *cmd;
    wmi_buf_t buf;
    int len = 0;
    int ret;
    len = sizeof(wmi_pdev_smart_ant_enable_cmd);
    buf = wmi_buf_alloc(scn->wmi_handle, len);
    if (!buf) {
        adf_os_print("%s:wmi_buf_alloc failed\n", __FUNCTION__);
        return ;
    }
    
    cmd = (wmi_pdev_smart_ant_enable_cmd *)wmi_buf_data(buf);
    cmd->enable = enable;
    cmd->mode = mode;
    cmd->rx_antenna = rx_antenna;
    cmd->tx_default_antenna = rx_antenna;
    if (mode == SMART_ANT_MODE_SERIAL) {
        cmd->gpio_pin[0] = ATH_GPIOPIN_ANT_SERIAL_STROBE_RADIO1;
        cmd->gpio_pin[1] = ATH_GPIOPIN_ANT_SERIAL_DATA_RADIO1;
        cmd->gpio_pin[2] = 0;
        cmd->gpio_pin[3] = 0;

        cmd->gpio_func[0] = ATH_GPIOFUNC_ANT_SERIAL_STROBE_RADIO1;
        cmd->gpio_func[1] = ATH_GPIOFUNC_ANT_SERIAL_DATA_RADIO1;
        cmd->gpio_func[2] = 0;
        cmd->gpio_func[3] = 0;
    } else if (mode == SMART_ANT_MODE_PARALLEL){
        cmd->gpio_pin[0] = ATH_GPIOPIN_ANTCHAIN0_RADIO1;
        cmd->gpio_pin[1] = ATH_GPIOPIN_ANTCHAIN1_RADIO1;
        cmd->gpio_pin[2] = ATH_GPIOPIN_ANTCHAIN2_RADIO1;
        cmd->gpio_pin[3] = 0;

        cmd->gpio_func[0] = ATH_GPIOFUNC_ANTCHAIN0_RADIO1;
        cmd->gpio_func[1] = ATH_GPIOFUNC_ANTCHAIN1_RADIO1;
        cmd->gpio_func[2] = ATH_GPIOFUNC_ANTCHAIN2_RADIO1;
        cmd->gpio_func[3] = 0;
    }

    ret = wmi_unified_cmd_send(scn->wmi_handle,
                               buf,
                               len,
                               WMI_PDEV_SMART_ANT_ENABLE_CMDID);

    adf_os_print("%s: Sent WMI_PDEV_SMART_ANT_ENABLE_CMDID.\n"
            "enable:%d mode:%d  rx_antenna: 0x%08x PINS: [%d %d %d %d] Func[%d %d %d %d] cmdstatus=%d\n",
            __FUNCTION__,
            cmd->enable,
            cmd->mode,
            cmd->rx_antenna,
            cmd->gpio_pin[0], cmd->gpio_pin[1], cmd->gpio_pin[2], cmd->gpio_pin[3],
            cmd->gpio_func[0], cmd->gpio_func[1], cmd->gpio_func[2], cmd->gpio_func[3],
            ret);

    /* Enable txfeed back to receive TX Control and Status descriptors from target */    
    if( ret == 0) {
        if (enable) {
            ol_ath_smart_ant_enable_txfeedback(ic, 1);
        } else {
            ol_ath_smart_ant_enable_txfeedback(ic, 0);
        }
    } else {
        adf_os_print(" %s :WMI Failed \n", __func__);
    }
}

void ol_ath_smart_ant_set_rx_antenna(struct ieee80211com *ic, u_int32_t antenna)
{

    struct ol_ath_softc_net80211 *scn = OL_ATH_SOFTC_NET80211(ic);
    wmi_pdev_smart_ant_set_rx_antenna_cmd *cmd;
    wmi_buf_t buf;
    int len = 0;
    int ret;

    len = sizeof(wmi_pdev_smart_ant_set_rx_antenna_cmd);
    buf = wmi_buf_alloc(scn->wmi_handle, len);
    if (!buf) {
        adf_os_print("%s:wmi_buf_alloc failed\n", __FUNCTION__);
        return ; 
    }

    cmd = (wmi_pdev_smart_ant_set_rx_antenna_cmd *)wmi_buf_data(buf);
    cmd->rx_antenna = antenna;
    ret = wmi_unified_cmd_send(scn->wmi_handle,
                               buf,
                               len,
                               WMI_PDEV_SMART_ANT_SET_RX_ANTENNA_CMDID);

    adf_os_print("%s: Sent WMI_PDEV_SMART_ANT_SET_RX_ANTENNA_CMDID. \n"
            " rx_antenna: 0x%08x cmdstatus=%d\n",
            __FUNCTION__,
            cmd->rx_antenna,
            ret);

}
/* 
* TODO: As of now both RX antenna and TX default antenna are same.
* So call RX antena function and in future if required need to add a WMI for the same.
*/
void ol_ath_smart_ant_set_tx_default_antenna(struct ieee80211com *ic, u_int32_t antenna)
{
    ol_ath_smart_ant_set_rx_antenna(ic, antenna);
}

void ol_ath_smart_ant_set_tx_antenna(struct ieee80211_node *ni, u_int32_t *antenna_array)
{
    struct ieee80211com *ic = ni->ni_ic;
    struct ol_ath_softc_net80211 *scn = OL_ATH_SOFTC_NET80211(ic);
    struct ol_ath_vap_net80211 *avn = OL_ATH_VAP_NET80211(ni->ni_vap);
    wmi_peer_sant_set_tx_antenna_cmd *cmd;
    wmi_buf_t buf;
    int len = 0;
    int ret;

    len = sizeof(wmi_peer_sant_set_tx_antenna_cmd);
    buf = wmi_buf_alloc(scn->wmi_handle, len);
    if (!buf) {
        adf_os_print("%s:wmi_buf_alloc failed\n", __FUNCTION__);
        return ; 
    }

    cmd = (wmi_peer_sant_set_tx_antenna_cmd *)wmi_buf_data(buf);
    cmd->vdev_id = avn->av_if_id;
    WMI_CHAR_ARRAY_TO_MAC_ADDR(ni->ni_macaddr, &cmd->peer_macaddr);

    cmd->antenna_series[0] = antenna_array[0];
    cmd->antenna_series[1] = antenna_array[1];
    ret = wmi_unified_cmd_send(scn->wmi_handle,
                               buf,
                               len,
                               WMI_PEER_SMART_ANT_SET_TX_ANTENNA_CMDID);
   
    adf_os_print("%s: Sent WMI_PEER_SMART_ANT_SET_TX_ANTENNA_CMDID. \n"
            " Node: %s tx_antennas: [0x%08x 0x%08x] cmdstatus=%d\n",
            __FUNCTION__,
            ether_sprintf(ni->ni_macaddr),
            cmd->antenna_series[0],
            cmd->antenna_series[1],
            ret);


}

void ol_ath_smart_ant_set_training_info(struct ieee80211_node *ni, uint32_t *rate_array, uint32_t *antenna_array, uint32_t numpkts)
{
    struct ieee80211com *ic = ni->ni_ic;
    struct ol_ath_softc_net80211 *scn = OL_ATH_SOFTC_NET80211(ic);
    struct ol_ath_vap_net80211 *avn = OL_ATH_VAP_NET80211(ni->ni_vap);
    wmi_peer_sant_set_train_antenna_cmd *cmd;
    wmi_buf_t buf;
    int len = 0;
    int ret;

    len = sizeof(wmi_peer_sant_set_train_antenna_cmd);
    buf = wmi_buf_alloc(scn->wmi_handle, len);
    if (!buf) {
        adf_os_print("%s:wmi_buf_alloc failed\n", __FUNCTION__);
        return ; 
    }

    cmd = (wmi_peer_sant_set_train_antenna_cmd *)wmi_buf_data(buf);
    cmd->vdev_id = avn->av_if_id;
    WMI_CHAR_ARRAY_TO_MAC_ADDR(ni->ni_macaddr, &cmd->peer_macaddr);
    OS_MEMCPY(&cmd->train_rate_series[0], &rate_array[0], (sizeof(uint32_t)*SMART_ANT_MAX_RATE_SERIES));
    OS_MEMCPY(& cmd->train_antenna_series[0], &antenna_array[0], (sizeof(uint32_t)*SMART_ANT_MAX_RATE_SERIES));
    cmd->num_pkts = numpkts;
    ret = wmi_unified_cmd_send(scn->wmi_handle,
                               buf,
                               len,
                               WMI_PEER_SMART_ANT_SET_TRAIN_INFO_CMDID);

    adf_os_print("%s: Sent WMI_PEER_SMART_ANT_SET_TRAIN_INFO_CMDID. \n"
            " Train Node: %s rate_array[0x%02x 0x%02x] tx_antennas: [0x%08x 0x%08x] cmdstatus=%d \n",
            __FUNCTION__,
            ether_sprintf(ni->ni_macaddr),
            cmd->train_rate_series[0], cmd->train_rate_series[1],
            cmd->train_antenna_series[0],cmd->train_antenna_series[1],
            ret);


}

int ol_ath_smart_ant_rxfeedback(struct ol_txrx_pdev_t *pdev, struct ol_txrx_peer_t *peer, struct  sa_rx_feedback *rx_feedback)
{
    struct ol_txrx_vdev_t *vdev = NULL;
    struct ieee80211vap *vap = NULL;
    struct ieee80211_node *ni = NULL;
    int status = -1;
    struct ol_ath_softc_net80211 *scn =
        (struct ol_ath_softc_net80211 *)pdev->ctrl_pdev;

    vdev = peer->vdev;
    vap = ol_ath_vap_get(scn, vdev->vdev_id);
    ni = ieee80211_find_node(&vap->iv_ic->ic_sta,
            peer->mac_addr.raw);
    if (ni) {
        ieee80211_smart_ant_update_rxfeedback(ni, rx_feedback, status);
        ieee80211_free_node(ni);
    }
    return status;
}

A_STATUS
process_txfeedback(struct ol_txrx_pdev_t *txrx_pdev,
                    void *data, uint16_t peer_id,
                    enum htt_tx_status status)
{
    struct ath_smart_ant_pktlog_hdr pl_hdr;
    uint32_t *pl_tgt_hdr;
    int txstatus = 0;
    int i = 0;
    struct sa_tx_feedback tx_feedback;
    struct ol_ath_softc_net80211 *scn =
                    (struct ol_ath_softc_net80211 *)txrx_pdev->ctrl_pdev;

    struct ol_txrx_vdev_t *vdev = NULL;
    struct ol_txrx_peer_t *peer;
    struct ieee80211vap *vap = NULL;
    struct ieee80211_node *ni = NULL;

    if (!txrx_pdev) {
        adf_os_print("Invalid pdev in %s\n", __func__);
        return A_ERROR;
    }

    pl_tgt_hdr = (uint32_t *)data;
    pl_hdr.log_type =  (*(pl_tgt_hdr + ATH_SMART_ANT_PKTLOG_HDR_LOG_TYPE_OFFSET) &
                                        ATH_SMART_ANT_PKTLOG_HDR_LOG_TYPE_MASK) >>
                                        ATH_SMART_ANT_PKTLOG_HDR_LOG_TYPE_SHIFT;

    if ((pl_hdr.log_type == SMART_ANT_PKTLOG_TYPE_TX_CTRL)) {
        int frame_type;
        int peer_id;
        void *tx_ppdu_ctrl_desc;
        u_int32_t *tx_ctrl_ppdu, try_status = 0;
        uint8_t rt0, rt1, rt2, rt3, total_tries =0, sbw_indx_succ = 0;
        tx_ppdu_ctrl_desc = (void *)data + sizeof(struct ath_smart_ant_pktlog_hdr);

        tx_ctrl_ppdu = (u_int32_t *)tx_ppdu_ctrl_desc;
       
        peer_id = tx_ctrl_ppdu[TX_PEER_ID_OFFSET];

        frame_type = (tx_ctrl_ppdu[TX_FRAME_OFFSET]
                          & TX_FRAME_TYPE_MASK) >> TX_FRAME_TYPE_SHIFT;
      
        if (frame_type == TX_FRAME_TYPE_DATA) { /* data frame */

            if (g_tx_ppdu_end[SMART_ANT_FEEDBACK_OFFSET] == 0) {
                return A_ERROR;
            }

            peer = (peer_id == HTT_INVALID_PEER) ?
                NULL : txrx_pdev->peer_id_to_obj_map[peer_id];
            if (peer && !(peer->bss_peer)) {
                vdev = peer->vdev;
                vap = ol_ath_vap_get(scn, vdev->vdev_id);
                ni = ieee80211_find_node(&vap->iv_ic->ic_sta,
                        peer->mac_addr.raw);
              
                if (!ni) {
                    return A_ERROR;
                }
                
                total_tries = (g_tx_ppdu_end[TX_TOTAL_TRIES_OFFSET] & TX_TOTAL_TRIES_MASK) >> TX_TOTAL_TRIES_SHIFT;

                OS_MEMZERO(&tx_feedback, sizeof(tx_feedback));
                tx_feedback.nPackets = (g_tx_ppdu_end[SMART_ANT_FEEDBACK_OFFSET] & 0xffff);
                tx_feedback.nBad = (g_tx_ppdu_end[SMART_ANT_FEEDBACK_OFFSET] & 0x7fff0000) >> 16;

                /* Rate code and Antenna values */
                tx_feedback.tx_antenna[0] = (tx_ctrl_ppdu[TX_ANT_OFFSET_S0] & TX_ANT_MASK);
                tx_feedback.tx_antenna[1] = (tx_ctrl_ppdu[TX_ANT_OFFSET_S1] & TX_ANT_MASK);
                
                /* RateCode */
                rt0 = ((tx_ctrl_ppdu[TXCTRL_S0_RATE_BW20_OFFSET] & TXCTRL_RATE_MASK) >> TXCTRL_RATE_SHIFT);
                rt1 = ((tx_ctrl_ppdu[TXCTRL_S0_RATE_BW40_OFFSET] & TXCTRL_RATE_MASK) >> TXCTRL_RATE_SHIFT);
                rt2 = ((tx_ctrl_ppdu[TXCTRL_S0_RATE_BW80_OFFSET] & TXCTRL_RATE_MASK) >> TXCTRL_RATE_SHIFT);
                rt3 = ((tx_ctrl_ppdu[TXCTRL_S0_RATE_BW160_OFFSET] & TXCTRL_RATE_MASK) >> TXCTRL_RATE_SHIFT);

                tx_feedback.rate_mcs[0] = ((rt3 << 24) | (rt2 << 16 )| (rt1 << 8) | rt0);

                rt0 = ((tx_ctrl_ppdu[TXCTRL_S1_RATE_BW20_OFFSET] & TXCTRL_RATE_MASK) >> TXCTRL_RATE_SHIFT);
                rt1 = ((tx_ctrl_ppdu[TXCTRL_S1_RATE_BW40_OFFSET] & TXCTRL_RATE_MASK) >> TXCTRL_RATE_SHIFT);
                rt2 = ((tx_ctrl_ppdu[TXCTRL_S1_RATE_BW80_OFFSET] & TXCTRL_RATE_MASK) >> TXCTRL_RATE_SHIFT);
                rt3 = ((tx_ctrl_ppdu[TXCTRL_S1_RATE_BW160_OFFSET] & TXCTRL_RATE_MASK) >> TXCTRL_RATE_SHIFT);

                tx_feedback.rate_mcs[1] = ((rt3 << 24) | (rt2 << 16 )| (rt1 << 8) | rt0);
                
                tx_feedback.rate_mcs[2] = 0;
                tx_feedback.rate_mcs[3] = 0;

                /* Extract and fill */
                /* index0 - s0_bw20, index1 - s0_bw40  index4 - s1_bw20 ... index7: s1_bw160 */
                for (i = 0; i < MAX_RETRIES; i++) {
                    tx_feedback.nlong_retries[i] =  ((g_tx_ppdu_end[LONG_RETRIES_OFFSET] >> (i*4)) & 0x0f);
                    tx_feedback.nshort_retries[i] = ((g_tx_ppdu_end[SHORT_RETRIES_OFFSET] >> (i*4)) & 0x0f);

                    /* HW gives try counts and for SA module we need to provide failure counts
                     * So manipulate short failure count accordingly.
                     */
                    if (tx_feedback.nlong_retries[i]) {
                        if (tx_feedback.nshort_retries[i] == tx_feedback.nlong_retries[i]) {
                            tx_feedback.nshort_retries[i]--;
                        }
                    }
                }

                /* ACK RSSI */
                tx_feedback.rssi[0] = g_tx_ppdu_end[ACK_RSSI0_OFFSET];
                tx_feedback.rssi[1] = g_tx_ppdu_end[ACK_RSSI1_OFFSET];
                tx_feedback.rssi[2] = g_tx_ppdu_end[ACK_RSSI2_OFFSET];
                tx_feedback.rssi[3] = g_tx_ppdu_end[ACK_RSSI3_OFFSET];

                try_status = g_tx_ppdu_end[total_tries-1];
                sbw_indx_succ = (try_status & TX_TRY_SERIES_MASK)?NUM_DYN_BW_MAX:0;
                sbw_indx_succ += ((try_status & TX_TRY_BW_MASK) >> TX_TRY_BW_SHIFT);
                if (tx_feedback.nPackets != tx_feedback.nBad) {

                    if (tx_feedback.nlong_retries[sbw_indx_succ]) {
                        tx_feedback.nlong_retries[sbw_indx_succ] -= 1;
                    }

                    if (tx_feedback.nshort_retries[sbw_indx_succ]) {
                        tx_feedback.nshort_retries[sbw_indx_succ] -= 1;
                    }
                }

                tx_feedback.rate_index = sbw_indx_succ;
                tx_feedback.is_trainpkt = ((g_tx_ppdu_end[SMART_ANT_FEEDBACK_OFFSET] & SMART_ANT_FEEDBACK_TRAIN_MASK) ? 1: 0);

                /* Data recevied from the associated node, Prepare TX feed back structure and send to SA module */ 
                ieee80211_smart_ant_update_txfeedback(ni, &tx_feedback, txstatus);
                ieee80211_free_node(ni);
            }
        }
    }

    /* First We will get status */
    if (pl_hdr.log_type == SMART_ANT_PKTLOG_TYPE_TX_STAT) {
        void *tx_ppdu_status_desc;
        u_int32_t *tx_status_ppdu;
        tx_ppdu_status_desc = (void *)data + sizeof(struct ath_smart_ant_pktlog_hdr);
        tx_status_ppdu = (u_int32_t *)tx_ppdu_status_desc;
        /* cache ppdu end (tx status desc) for smart antenna txfeedback */
        OS_MEMCPY(&g_tx_ppdu_end, tx_status_ppdu, (sizeof(uint32_t)*MAX_TX_PPDU_SIZE));
    }
       
    return A_OK;
}

int
ol_ath_smart_ant_get_txfeedback(void *pdev, enum WDI_EVENT event,
                        void *log_data, uint16_t peer_id,
                        enum htt_rx_status status)
{
    switch(event) {
    case WDI_EVENT_TX_STATUS:
        process_txfeedback(pdev, log_data, peer_id, status);
        break;
    default:
        adf_os_print("%s: Un Subscribed Event: %d \n", __func__, event);
        break;
    }
    return 0;
}

int ol_ath_smart_ant_enable_txfeedback(struct ieee80211com *ic, int enable)
{
    struct ol_ath_softc_net80211 *scn = OL_ATH_SOFTC_NET80211(ic);
    uint32_t types=0;
    int len = 0;
    wmi_buf_t buf;
    wmi_pdev_pktlog_enable_cmd *cmd;

    if (enable == 1) {
        /* Call back for txfeedback */
        SMART_ANT_TX_SUBSCRIBER.callback = ol_ath_smart_ant_get_txfeedback;
        if(wdi_event_sub(scn->pdev_txrx_handle,
                        &SMART_ANT_TX_SUBSCRIBER,
                        WDI_EVENT_TX_STATUS)) {
            return A_ERROR;
        }
        types |= WMI_PKTLOG_EVENT_TX;
        len = sizeof(wmi_pdev_pktlog_enable_cmd);
        buf = wmi_buf_alloc(scn->wmi_handle, len);
        if (!buf) {
                adf_os_print("%s:wmi_buf_alloc failed\n", __FUNCTION__);
            return A_ERROR;
        }
        cmd = (wmi_pdev_pktlog_enable_cmd *)wmi_buf_data(buf);
        cmd->evlist = WMI_PKTLOG_EVENT_TX;
        /*enabling the pktlog for smart antenna tx feedback*/
        if(wmi_unified_cmd_send(scn->wmi_handle, buf, len,
                                WMI_PDEV_PKTLOG_ENABLE_CMDID)) {
            return A_ERROR;
        }
        return A_OK;
    } else if (enable == 0) {
        if(wdi_event_unsub(
                    scn->pdev_txrx_handle, 
                    &SMART_ANT_TX_SUBSCRIBER,
                    WDI_EVENT_TX_STATUS)) {
            return A_ERROR;
        }
        
        buf = wmi_buf_alloc(scn->wmi_handle, 0);
        if (!buf) {
            adf_os_print("%s:wmi_buf_alloc failed\n", __FUNCTION__);
            return A_ERROR;
        }
        if(!wmi_unified_cmd_send(scn->wmi_handle, buf, len,
                                WMI_PDEV_PKTLOG_DISABLE_CMDID)) {
            return A_ERROR;
        }
        return A_OK;
    } else {
        return A_ERROR;
    }
}

int ol_smart_ant_get_enable(struct ol_ath_softc_net80211 *scn)
{
    struct ieee80211com *ic = &scn->sc_ic;
    return ic->smart_ant_enable;
}
EXPORT_SYMBOL(ol_smart_ant_get_enable);

void ol_ath_smart_ant_attach(struct ieee80211com *ic)
{
    struct ol_ath_softc_net80211 *scn = OL_ATH_SOFTC_NET80211(ic);
    if (WMI_SERVICE_IS_ENABLED(scn->wmi_service_bitmap, WMI_SERVICE_SMART_ANTENNA)) {
        ic->ic_smart_ant_enable = ol_ath_smart_ant_enable;
        ic->ic_smart_ant_set_rx_antenna = ol_ath_smart_ant_set_rx_antenna;
        ic->ic_smart_ant_set_tx_antenna = ol_ath_smart_ant_set_tx_antenna;
        ic->ic_smart_ant_set_tx_default_antenna = ol_ath_smart_ant_set_tx_default_antenna;
        ic->ic_smart_ant_set_training_info = ol_ath_smart_ant_set_training_info;
        ic->ic_smart_ant_prepare_rateset = NULL;
        ic->max_fallback_rates = 1; /* 1 primary and 1 fall back rate */ 
        ic->radio_id = 1; /* Radio Id is 1 for 5Ghz/offload */
        ic->ic_smart_ant_state = SMART_ANT_STATE_DEFAULT;
        wmi_unified_register_event_handler(scn->wmi_handle, WMI_PEER_RATECODE_LIST_EVENTID,
                wmi_unified_smart_ant_assoc_handler, NULL);
    } else {
        ic->ic_smart_ant_enable = NULL;
        ic->ic_smart_ant_set_rx_antenna = NULL;
        ic->ic_smart_ant_set_tx_antenna = NULL;
        ic->ic_smart_ant_set_tx_default_antenna = NULL;
        ic->ic_smart_ant_set_training_info = NULL;
        ic->ic_smart_ant_prepare_rateset = NULL;
        adf_os_print("%s: Firmware doest not support Smart Antenna API. \n", __func__);
    }
}

void ol_ath_smart_ant_detach(struct ieee80211com *ic)
{
    struct ol_ath_softc_net80211 *scn = OL_ATH_SOFTC_NET80211(ic);
    if (WMI_SERVICE_IS_ENABLED(scn->wmi_service_bitmap, WMI_SERVICE_SMART_ANTENNA)) {
        wmi_unified_unregister_event_handler(scn->wmi_handle, WMI_PEER_RATECODE_LIST_EVENTID);
    }
}

#endif
