/*
 *  Copyright (c) 2009 Atheros Communications Inc.  All rights reserved.
 */


#ifndef _IEEE80211_RRM_PROTO_H_
#define _IEEE80211_RRM_PROTO_H_ 

#define IEEE80211_ACTION_CAT_RM    5   /* Radio measurement */
#if UMAC_SUPPORT_RRM

#define IEEE80211_ELEMID_NEIGHBOR_REPORT          52

/* rrm call back struct */
struct ieee80211_rrm_cbinfo {
    u_int8_t* frm;
    u_int8_t* ssid;
    u_int8_t  ssid_len;
    u_int8_t  bssid[6];
    u_int16_t duration;
    u_int16_t chnum;
};

#define SLIDING_WINDOW   120
#define GRANUALITY_TIMER  1
#define STATS_TIMER       10
#define SLIDING_WINDOW_SIZE SLIDING_WINDOW/STATS_TIMER
#define RSSI_CLASS_MAX 7
#define TIME_SEC_TO_MSEC(sec)   (sec*1000)

struct ieee80211_beacon_report {
    u_int8_t reg_class;
    u_int8_t ch_num;
    u_int8_t ms_time[8];
    u_int16_t mduration;
    u_int8_t frame_info;
    u_int8_t rcpi;
    u_int8_t rsni;
    u_int8_t bssid[6];
    u_int8_t ant_id;
    u_int8_t parent_tsf[4];
    u_int8_t ies[1];
}__packed;

struct ieee80211_beacon_entry {
    TAILQ_ENTRY(ieee80211_beacon_entry) blist;
    struct ieee80211_beacon_report report;
};

struct ieee80211_beacon_report_table
{
    spinlock_t                              lock;     /* on scan table */
    TAILQ_HEAD(, ieee80211_beacon_entry)    entry;    /* head all entries */
};

struct vodelay
{
    u_int8_t  sliding_vo[SLIDING_WINDOW_SIZE];
    u_int8_t  min;
    u_int8_t  max;
    u_int8_t  avg;
    u_int8_t  last;
    u_int8_t index;
    u_int8_t valid_entries;
};
struct anpi
{
    int8_t  sliding_anpi[SLIDING_WINDOW_SIZE];
    int8_t  min;
    int8_t  max;
    int8_t  avg;
    int8_t  last;
    int8_t index;
    int8_t valid_entries;
};

struct bedelay
{
    u_int8_t  sliding_be[SLIDING_WINDOW_SIZE];
    u_int8_t  min;
    u_int8_t  max;
    u_int8_t  avg;
    u_int8_t  last;
    u_int8_t index;
    u_int8_t valid_entries;
};

struct frmcnt
{
    u_int32_t  sliding_frmcnt[SLIDING_WINDOW_SIZE];
    u_int32_t  min;
    u_int32_t  max;
    u_int32_t  avg;
    u_int32_t  last;
    u_int32_t index;
    u_int32_t valid_entries;
};
struct ackfail
{
    u_int32_t  sliding_ackfail[SLIDING_WINDOW_SIZE];
    u_int32_t  min;
    u_int32_t  max;
    u_int32_t  avg;
    u_int32_t  last;
    u_int32_t index;
    u_int32_t valid_entries;
};
struct per
{
    u_int32_t  min;
    u_int32_t  max;
    u_int32_t  avg;
    u_int32_t  last;
};
struct stcnt{
    u_int8_t  sliding_stcnt[SLIDING_WINDOW_SIZE];
    u_int8_t  min;
    u_int8_t  max;
    u_int8_t  avg;
    u_int8_t  last;
    u_int8_t index;
    u_int8_t valid_entries;
};
struct chload
{
    u_int8_t  sliding_chload[SLIDING_WINDOW_SIZE];
    u_int8_t  min;
    u_int8_t  max;
    u_int8_t  avg;
    u_int8_t  last;
    u_int8_t index;
    u_int8_t valid_entries;
    u_int8_t pre_trap_val;
};

struct sliding_window
{
    struct chload    chload_window;
    struct bedelay   be_window;
    struct vodelay   vo_window;
    struct anpi      anpi_window;
    struct frmcnt    frmcnt_window;
    struct ackfail   ackfail_window;
    struct per       per_window;
    struct stcnt     stcnt_window;
};

typedef struct ieee80211_rrm
{
    u_int32_t                           rrm_in_progress; /* to know whether rrm is busy or not*/
    u_int32_t                           rrm_last_scan;/* last scan time */
    IEEE80211_SCAN_REQUESTOR            rrm_scan_requestor;
    IEEE80211_SCAN_ID                   rrm_scan_id;     /* scan id assigned by scan scheduler */
    osdev_t                             rrm_osdev;
    wlan_if_t                           rrm_vap;/* vap struct */
    struct ieee80211_node               *ni;/* node */
    int8_t                              rrm_noisefloor[IEEE80211_RRM_CHAN_MAX]; /* calulation of noise */
    int8_t                              u_chload[IEEE80211_RRM_CHAN_MAX]; /*channl load storage to feed to user layer */
    ieee80211_rrm_noise_data_t          user_nhist_resp[IEEE80211_RRM_CHAN_MAX]; /* user layer feed after report received on ap side */
    u_int32_t                           rrm_chan_load[IEEE80211_RRM_CHAN_MAX];  /*channel load calculation through scan module */
    u_int32_t                           rrm_cycle_count[IEEE80211_RRM_CHAN_MAX];
    u_int8_t                            pending_report;
    u_int8_t                            pending_report_type;
    void                                *rrmcb;
    u_int8_t                            req_chan;
    os_timer_t                          rrm_sliding_window_timer;
    uint8_t                             pre_gid; /*required in case of sta stats in timer */
    adf_os_work_t                       rrm_work;
    struct sliding_window               window; /* window to mantain stats  */
    atomic_t                            timer_running;
    atomic_t                            rrm_detach;
    u_int8_t                            windowstats; /* following varaibles are for sending various traps from work queue */
    u_int8_t                            nonerptrap;
    u_int8_t                            bgjointrap;
    u_int8_t                            chloadtrap;
    u_int32_t                           rrmstats;
    u_int32_t                           slwindow;
    u_int8_t                            cochannel;
    u_int8_t                            precochannel;
    u_int8_t                            cochanneltrap;
    struct ieee80211_beacon_report_table *beacon_table;
#if UMAC_SUPPORT_RRM_DEBUG
    u_int32_t                           rrm_dbg_lvl;
#endif
}*ieee80211_rrm_t;


/**
 * rrm action frame recv handler
 * @param vaphandle         : handle to the VAP
 * @param ni                : handle to node pointer
 * @param action            : rrm action type 
 * @param frm               : pointer to action frame 
 * @param frm_len           : action frame length 
 *
 * @return on success return 0.
 *         on failure returns -ve value.
 */
int ieee80211_rrm_recv_action(wlan_if_t vap, wlan_node_t ni, 
                            u_int8_t action, u_int8_t *frm, u_int32_t frm_len);
/**
 * Add RRM capability IE
 * @param frm               : pointer to frame 
 * @param ni                : handle to node pointer
 *
 * @return : pointer to frame after adding ie
 */
u_int8_t *ieee80211_add_rrm_cap_ie(u_int8_t *frm, struct ieee80211_node *ni);

/**
 * Go through the scan list and fill 
 * beacon request response. 
 * @param arg               : opaque pointer to ieee80211_rrm_cbinfo
 * @param se                : scan entry 
 *
 * @return on success return 0.
 *         on failure returns -ve value.
 */
int ieee80211_fill_bcnreq_info(void *arg, wlan_scan_entry_t se);
/**
 * Go through the scan list and fill the neighbor
 * AP's information
 * @param arg               : opaque pointer to ieee80211_rrm_cbinfo
 * @param se                : scan entry 
 *
 * @return on success return 0.
 *         on failure returns -ve value.
 */
int ieee80211_fill_nrinfo(void *arg, wlan_scan_entry_t se);
/**
 * Attach function for rrm module 
 * param ic               : pointer to ic structure 
 * param vap              :pointer to vap structure 
 * @return on success return 0.
 *         on failure returns -ve value.
 */
int ieee80211_rrm_vattach (struct ieee80211com *ic,ieee80211_vap_t vap); 

/**
 * Detach  function for rrm module 
 * param vap              :pointer to vap structure 
 * @return on success return 0.
 *         on failure returns -ve value.
 */

int ieee80211_rrm_vdetach(ieee80211_vap_t vap);

/**
 *  Go through all node list and find node is rrm 
 *  capable or not.
 * param vap:pointer to vap structure 
 * param mac:pointer to mac address of node 
 * @return on success non zero value .
 *         on failure returns zero value.
 */
int ieee80211_node_is_rrm(wlan_if_t vap,char *mac);

/**
 * mark node as rrm capable  
 * param ni:pointer node structure
 * param flag:enable or disable 
 * @return :- void 
 *   
 */

void ieee80211_set_node_rrm(struct ieee80211_node *ni,uint8_t flag);

/* set rrmstats :- if enabled we should send various events to 
 * upper application(athadhoc) *
 */

#else
#define ieee80211_recv_link_measurement_req(_ni,_wbuf,_subtype)  (0)

#define ieee80211_recv_link_measurement_rsp(_ni,_wbuf,_subtype)  (0)

#define ieee80211_recv_neighbor_req(_ni, _wbuf, _subtype)        (0)

#define ieee80211_recv_neighbor_rsp(_ni,_wbuf,_subtype)          (0)

#define ieee80211_rrm_vattach(ic, vap) 

#define ieee80211_rrm_vdetach(vap) 

static inline u_int8_t *ieee80211_add_rrm_cap_ie(u_int8_t *frm, struct ieee80211_node *ni)
{ 
    return frm;
}
static inline int ieee80211_rrm_recv_action(wlan_if_t vap, wlan_node_t ni, 
                            u_int8_t action, u_int8_t *frm, u_int32_t frm_len)
{
     return EOK;
}

static inline void ieee80211_set_node_rrm(struct ieee80211_node *ni,uint8_t flag)
{
    return;
}
#endif /* UMAC_SUPPORT_RRM */

#if UMAC_SUPPORT_RRM_MISC
int ieee80211_set_rrmstats(wlan_if_t vap, int val);

/* get  rrmstats :- to get rrmstats value */
int ieee80211_get_rrmstats(wlan_if_t vap);

/* set sliding window  :- if enabled we should start internal sliding window */

int ieee80211_rrm_set_slwindow(wlan_if_t vap, int val);

/* to get vlaue of sliding window */
int ieee80211_rrm_get_slwindow(wlan_if_t vap);
#else
static inline int ieee80211_set_rrmstats(wlan_if_t vap, int val) 
{
    return EINVAL;
}
static inline int ieee80211_get_rrmstats(wlan_if_t vap) 
{
    return EINVAL;
}
static inline int ieee80211_rrm_set_slwindow(wlan_if_t vap, int val)
{
    return EINVAL;
}
static inline int ieee80211_rrm_get_slwindow(wlan_if_t vap) 
{ 
    return EINVAL;

}
#endif /*UMAC_SUPPORT_RRM_MISC*/

#if UMAC_SUPPORT_RRM_DEBUG
/**
 * @brief 
 *
 * @param vap
 * @param val
 *
 * @return 
 */
int ieee80211_rrmdbg_get(wlan_if_t vap);

/**
 * @brief 
 *
 * @param vap
 *
 * @return 
 */
int ieee80211_rrmdbg_set(wlan_if_t vap, u_int32_t val);
#else

/**
 * @brief 
 *
 * @param vap
 *
 * @return 
 */
static inline int ieee80211_rrmdbg_set(wlan_if_t vap, u_int32_t val) 
{
    return -EINVAL;
}

/**
 * @brief 
 *
 * @param vap
 *
 * @return 
 */
static inline int ieee80211_rrmdbg_get(wlan_if_t vap)
{
    return -EINVAL;
}
#endif /*UMAC_SUPPORT_RRM_DEBUG*/
#endif /* _IEEE80211_RRM_H_ */
