
/*
 *  Copyright (c) 2009 Atheros Communications Inc.  All rights reserved.
 */


/*
 *  Radio Resource measurements message handlers for Station .
 */

#include <ieee80211_var.h>
#include "ieee80211_rrm_priv.h"

#if UMAC_SUPPORT_RRM

/**
 * @brief  Routine to send reject frame
 *
 * @param rrm
 * @param rm_req
 *
 * @return on success return 0.
 *         on failure returns -ve value.
 */
int rrm_send_reject(ieee80211_rrm_t rrm,
       struct ieee80211_action_rm_req *rm_req )
{
    struct ieee80211_rrmreq_info *params;
    struct ieee80211_measreq_ie *req;
#if UMAC_SUPPORT_RRM_DEBUG 
    wlan_if_t vap = rrm->rrm_vap;
#endif        
    RRM_FUNCTION_ENTER;

    req = (struct ieee80211_measreq_ie *)(&(rm_req->req_ies[0]));
    params = (struct ieee80211_rrmreq_info *)
               OS_MALLOC(rrm->rrm_osdev, sizeof(struct ieee80211_rrmreq_info), 0);

    if(NULL == params)
        return -EBUSY;

    params->rm_dialogtoken = rm_req->dialogtoken;
    params->rep_dialogtoken = req->token ; 
    params->reject_type = req->reqtype; 
    params->reject_mode = BIT_REFUSED;

    ieee80211_rrm_set_report_pending(rrm->rrm_vap,IEEE80211_MEASREQ_OTHER,(void *)params);

    if(rrm->pending_report)
        ieee80211_send_report(rrm);

    RRM_FUNCTION_EXIT;

    return EOK;
}

/**
 * @brief routine to recv chload request 
 *
 * @param vap
 * @param ni
 * @param req
 * @param rm_token
 *
 * @return on success return 0.
 *         on failure returns -ve value.
 */

int ieee80211_rrm_recv_chload_req(wlan_if_t vap, wlan_node_t ni,
        struct ieee80211_measreq_ie *req,u_int8_t rm_token)
{
    struct ieee80211_chloadreq *chload;
    struct ieee80211_rrmreq_info *params;

    RRM_FUNCTION_ENTER;

    chload = (struct ieee80211_chloadreq *)(&req->req[0]);
    params = (struct ieee80211_rrmreq_info *)
        OS_MALLOC(vap->rrm->rrm_osdev, sizeof(struct ieee80211_rrmreq_info), 0);

    if(NULL == params)
        return -EBUSY;

    params->rm_dialogtoken = rm_token;
    params->rep_dialogtoken = req->token;
    params->duration = chload->mduration;
    params->chnum = chload->chnum;
    params->regclass = chload->regclass;
    RRM_DEBUG(vap, RRM_DEBUG_INFO, 
              "%s : duration %d chnum %d regclass %d\n", __func__,  
               params->duration, params->chnum, params->regclass);

    ieee80211_rrm_set_report_pending(vap,IEEE80211_MEASREQ_CHANNEL_LOAD_REQ,(void *)params);
    if (ieee80211_rrm_scan_start(vap, true) != 0)
    {
        ieee80211_rrm_free_report(vap->rrm);
        return -EBUSY;
    }

    RRM_FUNCTION_EXIT;
    return EOK;
}

/**
 * @brief Routine to accept station statistics request 
 *
 * @param vap
 * @param ni
 * @param req
 * @param rm_token
 *
 * @return  
 * @return on success return 0.
 *         on failure returns -ve value.
 */
int ieee80211_rrm_recv_stastats_req(wlan_if_t vap, wlan_node_t ni, 
                struct ieee80211_measreq_ie *req,u_int8_t rm_token)
{
    struct ieee80211_rrmreq_info *params;
    struct ieee80211_stastatsreq *statsreq;

    RRM_FUNCTION_ENTER;
    statsreq = (struct ieee80211_stastatsreq *)(&(req->req[0]));

    params = (struct ieee80211_rrmreq_info *)
            OS_MALLOC(vap->rrm->rrm_osdev, sizeof(struct ieee80211_rrmreq_info), 0);

    if(NULL == params)
        return -EBUSY;
    params->rm_dialogtoken = rm_token;
    params->rep_dialogtoken= req->token;
    params->duration=statsreq->mduration;
    params->gid=statsreq->gid;

    RRM_DEBUG(vap, RRM_DEBUG_INFO, 
              "%s : duration %d chnum %d gid %d\n", __func__,  
               params->duration, params->chnum, params->gid);

    IEEE80211_ADDR_COPY(params->bssid,statsreq->dstmac);
    ieee80211_rrm_set_report_pending(vap,IEEE80211_MEASREQ_STA_STATS_REQ,(void *)params);

    if(vap->rrm->pending_report)
        ieee80211_send_report(vap->rrm);

    RRM_FUNCTION_EXIT;
    return EOK;
}

/**
 * @brief Routine to process location request  
 *
 * @param vap
 * @param ni
 * @param req
 * @param rm_token
 *
 * @return  
 * @return on success return 0.
 *         on failure returns -ve value.
 */

int ieee80211_rrm_recv_lci_req(wlan_if_t vap, wlan_node_t ni,
                            struct ieee80211_measreq_ie *req, u_int8_t rm_token)
{
    struct ieee80211_rrmreq_info *params;
    struct ieee80211_lcireq *lcireq;

    RRM_FUNCTION_ENTER;

    lcireq = (struct ieee80211_lcireq *)(&(req->req[0]));

    params = (struct ieee80211_rrmreq_info *)
              OS_MALLOC(vap->rrm->rrm_osdev, sizeof(struct ieee80211_rrmreq_info), 0);

    if(NULL == params)
        return -EBUSY;

    params->rm_dialogtoken = rm_token;
    params->rep_dialogtoken = req->token;
    params->location = lcireq->location;
    params->lat_res = lcireq->lat_res;
    params->long_res = lcireq->long_res;
    params->alt_res = lcireq->alt_res;

    RRM_DEBUG(vap, RRM_DEBUG_INFO, 
              "%s : location %d lat_res %d long_res %d alt_res %d\n", __func__,  
               params->location, params->lat_res, params->long_res, params->alt_res);

    ieee80211_rrm_set_report_pending(vap, IEEE80211_MEASREQ_LCI_REQ, (void *)params);
 
    if(vap->rrm->pending_report)
        ieee80211_send_report(vap->rrm);
 
    RRM_FUNCTION_EXIT;
    return EOK;
}

/**
 * @brief 
 *
 * @param vap
 * @param mode
 *
 * @return 
 * @return on success return 0.
 *         on failure returns -ve value.
 */
int ieee80211_rrm_scan(wlan_if_t vap, u_int8_t mode)
{
    int retval = 0;

    RRM_FUNCTION_ENTER;

    if ((mode == 0) || (mode == 1))
    {
        /* Active mode */
        wlan_scan_table_flush(vap);
        retval = ieee80211_rrm_scan_start(vap, (mode == 1)?true:false);
        if ((retval != 0) && vap->rrm->pending_report)
        {
            ieee80211_rrm_free_report(vap->rrm);
            return -EBUSY;
        }
        else
        {
             vap->rrm->rrm_last_scan = OS_GET_TIMESTAMP();
        }
    }
    else
    {
        /* Send last Beacon Table as report */
        if(vap->rrm->pending_report)
        {
            ieee80211_send_report(vap->rrm);
            return EOK;
        }
    }

    RRM_FUNCTION_EXIT;

    return EOK;
}

/**
 * @brief 
 *
 * @param vap
 * @param ni
 * @param req
 * @param rm_token
 *
 * @return 
 * @return on success return 0.
 *         on failure returns -ve value.
 */
int ieee80211_rrm_recv_beacon_req(wlan_if_t vap, wlan_node_t ni, 
                struct ieee80211_measreq_ie *req,u_int8_t rm_token)
{
    struct ieee80211_rrmreq_info *params=NULL;
    struct ieee80211_beaconreq *bcnreq=NULL;

    RRM_FUNCTION_ENTER;


    bcnreq = (struct ieee80211_beaconreq *)(&(req->req[0]));
    params = (struct ieee80211_rrmreq_info *)
           OS_MALLOC(vap->rrm->rrm_osdev, sizeof(struct ieee80211_rrmreq_info), 0);

    if(NULL == params)
        return -EBUSY;
    params->rm_dialogtoken = rm_token;
    params->rep_dialogtoken= req->token;
    params->duration=bcnreq->duration;
    params->chnum = bcnreq->channum;
 
    RRM_DEBUG(vap, RRM_DEBUG_INFO, 
              "%s : duration %d chnum %d regclass %d\n", __func__,  
               params->duration, params->chnum, params->regclass);

    IEEE80211_ADDR_COPY(params->bssid,bcnreq->bssid);
    ieee80211_rrm_set_report_pending(vap,IEEE80211_MEASREQ_BR_TYPE,(void *)params);

    if(vap->rrm->rrm_last_scan == 0)
    {
        ieee80211_rrm_scan(vap, bcnreq->mode);
        return EOK;
    }
    else 
    {
        u_int32_t last_scan_time=0,msec_current_time = 0;
        last_scan_time = (u_int32_t) CONVERT_SYSTEM_TIME_TO_MS(vap->rrm->rrm_last_scan);
        msec_current_time = (u_int32_t) CONVERT_SYSTEM_TIME_TO_MS(OS_GET_TIMESTAMP());
        msec_current_time -= last_scan_time;
    
        if(msec_current_time <  60*1000)
        {
            if(vap->rrm->pending_report)
                ieee80211_send_report(vap->rrm);
        }
        else
        {
            ieee80211_rrm_scan(vap, bcnreq->mode);
        }
    }

    RRM_FUNCTION_EXIT;
    return EOK;
}

/**
 * @brief process noise histogram request  
 *
 * @param vap
 * @param ni
 * @param req
 * @param rm_token
 *
 * @return 
 * @return on success return 0.
 *         on failure returns -ve value.
 */
int ieee80211_rrm_recv_nhist_req(wlan_if_t vap, wlan_node_t ni, 
              struct ieee80211_measreq_ie *req,u_int8_t rm_token)
{
    struct ieee80211_nhistreq *nhist=NULL;
    struct ieee80211_rrmreq_info *params=NULL;

    RRM_FUNCTION_ENTER;

    nhist = (struct ieee80211_nhistreq *)(&req->req[0]);
    params = (struct ieee80211_rrmreq_info *)
               OS_MALLOC(vap->rrm->rrm_osdev, sizeof(struct ieee80211_rrmreq_info), 0);

    if(NULL == params)
        return -EBUSY;

    params->rm_dialogtoken = rm_token;
    params->rep_dialogtoken = req->token;
    params->duration = nhist->mduration;
    params->chnum = nhist->chnum;
    params->regclass = nhist->regclass;


    RRM_DEBUG(vap, RRM_DEBUG_INFO, 
              "%s : duration %d chnum %d regclass %d\n", __func__,  
               params->duration, params->chnum, params->regclass);

    ieee80211_rrm_set_report_pending(vap,IEEE80211_MEASREQ_NOISE_HISTOGRAM_REQ,(void *)params);

    if (ieee80211_rrm_scan_start(vap, true)) {
      ieee80211_rrm_free_report(vap->rrm);
      return -EBUSY;
    }

    RRM_FUNCTION_EXIT;
    return EOK;
}

/**
 * @brief send response to beacon request  
 *
 * @param rrm
 * @param ie
 *
 * @return 
 */
u_int8_t *ieee80211_rrm_send_bcnreq_resp(ieee80211_rrm_t rrm , u_int8_t *ie)
{
    struct ieee80211_rrmreq_info *params;
    struct ieee80211_rrm_cbinfo cb_info;
    struct ieee80211vap *vap =NULL;
    vap = rrm->rrm_vap;

    RRM_FUNCTION_ENTER;
    params = (struct ieee80211_rrmreq_info *)(rrm->rrmcb);
    cb_info.frm = ie;
    IEEE80211_ADDR_COPY(cb_info.bssid, params->bssid);
    cb_info.chnum = params->chnum; 
    cb_info.duration = params->duration;
    wlan_scan_table_iterate(vap, ieee80211_fill_bcnreq_info, &cb_info);

    RRM_FUNCTION_EXIT;
    return cb_info.frm;
}

/**
 * @brief  response to chload request  
 *
 * @param rrm
 * @param ie
 *
 * @return 
 */
u_int8_t *ieee80211_rrm_send_chload_resp(ieee80211_rrm_t rrm , u_int8_t *ie)
{
    struct ieee80211_chloadrsp *chloadrsp=NULL;
    struct ieee80211_measrsp_ie *measrsp=NULL;
    struct ieee80211_rrmreq_info *params=NULL;
#if UMAC_SUPPORT_RRM_DEBUG 
    wlan_if_t vap = rrm->rrm_vap;
#endif        

    RRM_FUNCTION_ENTER;

    params = (struct ieee80211_rrmreq_info *)(rrm->rrmcb);
    measrsp = (struct ieee80211_measrsp_ie *)ie;
    measrsp->id = IEEE80211_ELEMID_MEASREP;
    measrsp->token=params->rep_dialogtoken;
    measrsp->rspmode = 0x00; /* Need to validate */
    measrsp->rsptype = IEEE80211_MEASRSP_CHANNEL_LOAD_REPORT;
    chloadrsp = (struct ieee80211_chloadrsp *)&measrsp->rsp[0];
    chloadrsp->regclass = params->regclass; 
    chloadrsp->chnum =params->chnum;
    OS_GET_RANDOM_BYTES(&chloadrsp->tsf[0], sizeof(chloadrsp->tsf));
    chloadrsp->mduration= params->duration;
    chloadrsp->chload = rrm->rrm_chan_load[chloadrsp->chnum];
    ie = (u_int8_t *)(&chloadrsp->rsp[0]);
    measrsp->len = ie -(u_int8_t *)(&measrsp->token);

    RRM_FUNCTION_EXIT;
    return ie;
}

/**
 * @brief noise histogrm response 
 *
 * @param rrm
 * @param ie
 *
 * @return 
 */
u_int8_t *ieee80211_rrm_send_nhist_resp(ieee80211_rrm_t rrm,u_int8_t *ie)
{
    struct ieee80211_nhistrsp *rsp=NULL;
    struct ieee80211_measrsp_ie *measrsp=NULL;
    struct ieee80211_rrmreq_info *params=NULL;
    unsigned long tsf;
#if UMAC_SUPPORT_RRM_DEBUG 
    wlan_if_t vap = rrm->rrm_vap;
#endif        

    RRM_FUNCTION_ENTER;

    params = (struct ieee80211_rrmreq_info *)(rrm->rrmcb);
    measrsp = (struct ieee80211_measrsp_ie *)ie;
    measrsp->id = IEEE80211_ELEMID_MEASREP;
    measrsp->token=params->rep_dialogtoken;
    measrsp->rspmode = 0x00; /* Need to validate */
    measrsp->rsptype = IEEE80211_MEASRSP_NOISE_HISTOGRAM_REPORT;
    rsp = (struct ieee80211_nhistrsp *)(&measrsp->rsp[0]);
    rsp->mduration = params->duration;
    rsp->regclass = params->regclass;
    rsp->chnum = params->chnum;
    rsp->anpi = (u_int8_t)rrm->rrm_noisefloor[rsp->chnum];
    OS_GET_RANDOM_BYTES(&rsp->antid,sizeof(rsp->antid));
    OS_GET_RANDOM_BYTES(rsp->ipi,sizeof(rsp->ipi));
    tsf = OS_GET_TIMESTAMP();
    OS_MEMCPY(rsp->tsf,&tsf,4);
    ie = &rsp->rsp[0];
    measrsp->len = (ie - &measrsp->token);

    RRM_FUNCTION_EXIT;
    return ie;
}

/**
 * @brief location configuration response  
 *
 * @param rrm
 * @param ie
 *
 * @return ie
 */
u_int8_t *ieee80211_rrm_send_lci_resp(ieee80211_rrm_t rrm,u_int8_t *ie)
{
    struct ieee80211_lcirsp *lcirpt;
    struct ieee80211_measrsp_ie *measrsp;
    struct ieee80211_rrmreq_info *params;
#if UMAC_SUPPORT_RRM_DEBUG 
    wlan_if_t vap = rrm->rrm_vap;
#endif        

    RRM_FUNCTION_ENTER;

    params = (struct ieee80211_rrmreq_info *)(rrm->rrmcb);
    measrsp = (struct ieee80211_measrsp_ie *)ie;

    measrsp->id = IEEE80211_ELEMID_MEASREP;
    measrsp->token = params->rep_dialogtoken;
    measrsp->rspmode = 0x00; /* Need to validate */
    measrsp->rsptype = IEEE80211_MEASRSP_LCI_REPORT;

    lcirpt = (struct ieee80211_lcirsp *)(&measrsp->rsp[0]);
  
    OS_GET_RANDOM_BYTES(&lcirpt->lci_data, sizeof(lcirpt->lci_data));

    ie = &lcirpt->rsp[0];

    measrsp->len = (ie - &measrsp->token);

    RRM_FUNCTION_EXIT;
    return ie;
}

/**
 * @brief sta statistics response  
 *
 * @param rrm
 * @param ie
 *
 * @return 
 */
u_int8_t *ieee80211_rrm_send_stastats_resp(ieee80211_rrm_t rrm,u_int8_t *ie)
{
    struct ieee80211_stastatsrsp *rsp;
    struct ieee80211_measrsp_ie *measrsp;
    struct ieee80211_rrmreq_info *params;
    unsigned long random = 0;
#if UMAC_SUPPORT_RRM_DEBUG 
    wlan_if_t vap = rrm->rrm_vap;
#endif        
    RRM_FUNCTION_ENTER;

    params = (struct ieee80211_rrmreq_info *)(rrm->rrmcb);
    measrsp = (struct ieee80211_measrsp_ie *)ie;
    measrsp->id = IEEE80211_ELEMID_MEASREP;
    measrsp->token=params->rep_dialogtoken;
    measrsp->rspmode = 0x00; /* Need to validate */
    measrsp->rsptype = IEEE80211_MEASRSP_STA_STATS_REPORT;
    rsp = (struct ieee80211_stastatsrsp *)(&measrsp->rsp[0]);
    rsp->m_intvl = params->duration;
    rsp->gid = params->gid;
    switch(rsp->gid)
    {
        case IEEE80211_STASTATS_GID0:
            OS_GET_RANDOM_BYTES(&random,sizeof(random));
            rsp->stats.gid0.txfragcnt =random % 100; 
            rsp->stats.gid0.mcastrxfrmcnt = random % 23;
            rsp->stats.gid0.failcnt   = random %89;
            rsp->stats.gid0.rxfragcnt = random % 87;
            rsp->stats.gid0.fcserrcnt = random %13;
            rsp->stats.gid0.txfrmcnt  =  random % 32103;
            ie = &(rsp->rsp[0]);
            break;
        case IEEE80211_STASTATS_GID1:
            OS_GET_RANDOM_BYTES(&random,sizeof(random));
            rsp->stats.gid1.rty =random % 100; 
            rsp->stats.gid1.multirty = random % 23;
            rsp->stats.gid1.frmdup = random %33;
            rsp->stats.gid1.rtsuccess = random % 87;
            rsp->stats.gid1.rtsfail = random %13;
            rsp->stats.gid1.ackfail =  random % 29;
            ie = &(rsp->rsp[0]);
            break;
        case IEEE80211_STASTATS_GID10: /*Average access delay */
            OS_GET_RANDOM_BYTES(&random,sizeof(random));
            if(random)
            {  
                rsp->stats.gid10.ap_avg_delay  = random % 88; 
                rsp->stats.gid10.be_avg_delay  = random % 99;
                rsp->stats.gid10.bk_avg_delay  = random % 77;
            }
            OS_GET_RANDOM_BYTES(&random,sizeof(random)) ;
            if(random)
            {  
                rsp->stats.gid10.vi_avg_delay  = random % 47;
                rsp->stats.gid10.vo_avg_delay  = random % 34;
                rsp->stats.gid10.st_cnt = random % 128;
                rsp->stats.gid10.ch_util =random % 60;
            }
            ie = &(rsp->rsp[0]);
            break;
        default:
            break;
    }
    measrsp->len = (ie - &measrsp->token);

    RRM_FUNCTION_EXIT;
    return ie ;
}

/**
 * @brief 
 *
 * @param rrm
 */

void ieee80211_send_report(ieee80211_rrm_t rrm)
{
    struct ieee80211vap *vap =NULL;
    struct ieee80211_node *ni = NULL;
    struct ieee80211_measrsp_ie *measrsp=NULL;
    wbuf_t wbuf = NULL;
    u_int8_t *frm = NULL,*fptr = NULL;
    struct ieee80211_action_rm_rsp *rm_rsp=NULL;
    struct ieee80211_rrmreq_info *params=NULL;
    enum ieee80211_opmode opmode; 

    params = (struct ieee80211_rrmreq_info *)(rrm->rrmcb);
    vap = rrm->rrm_vap;
    opmode = wlan_vap_get_opmode(vap);
    ni = rrm->ni;

    RRM_FUNCTION_ENTER;

    wbuf = ieee80211_getmgtframe(ni,IEEE80211_FC0_SUBTYPE_ACTION, &frm,0);
    if(wbuf == NULL)
        return;

    rm_rsp = (struct ieee80211_action_rm_rsp *)(frm);
    rm_rsp->header.ia_category = IEEE80211_ACTION_CAT_RM;
    rm_rsp->header.ia_action  = IEEE80211_ACTION_RM_RESP; 
    rm_rsp->dialogtoken = params->rm_dialogtoken; 

    switch(rrm->pending_report_type)
    {
        case IEEE80211_MEASREQ_BR_TYPE:
            fptr = ieee80211_rrm_send_bcnreq_resp(rrm,&rm_rsp->rsp_ies[0]);
            break;
        case IEEE80211_MEASREQ_CHANNEL_LOAD_REQ:
            fptr = ieee80211_rrm_send_chload_resp(rrm,&rm_rsp->rsp_ies[0]);
            break;
        case IEEE80211_MEASREQ_STA_STATS_REQ:
            fptr = ieee80211_rrm_send_stastats_resp(rrm,&rm_rsp->rsp_ies[0]);
            break;
        case IEEE80211_MEASREQ_NOISE_HISTOGRAM_REQ:
            fptr = ieee80211_rrm_send_nhist_resp(rrm,&rm_rsp->rsp_ies[0]);
            break;
        case IEEE80211_MEASREQ_LCI_REQ:
            fptr = ieee80211_rrm_send_lci_resp(rrm, &rm_rsp->rsp_ies[0]);
            break;
        case IEEE80211_MEASREQ_OTHER:
            measrsp = (struct ieee80211_measrsp_ie *)(&rm_rsp->rsp_ies[0]);
            measrsp->id = IEEE80211_ELEMID_MEASREP;
            measrsp->token=params->rep_dialogtoken;
            measrsp->rspmode |= params->reject_mode;
            measrsp->rsptype = params->reject_type;
            fptr = (u_int8_t *)(&measrsp->rsp[0]);
            measrsp->len = fptr -(u_int8_t *)(&measrsp->token);
            break;
        default:
            return;
    }

    if(IEEE80211_ADDR_EQ(ni->ni_macaddr, vap->iv_myaddr)) { /* self report */
        ieee80211_rrm_recv_action(vap,ni,IEEE80211_ACTION_RM_RESP,frm,(fptr - frm));
        OS_FREE(wbuf);
    } else {
        wbuf_set_pktlen(wbuf, (fptr -(u_int8_t*)(wbuf_header(wbuf))));
        ieee80211_send_mgmt(vap, ni, wbuf, false);
    }
    rrm->rrm_in_progress = 0; /* making module available */
    ieee80211_rrm_free_report(rrm);

    RRM_FUNCTION_EXIT;
    return;
}

/**
 * @brief 
 *
 * @param vap
 * @param ni
 * @param rm_req
 * @param frm_len
 *
 * @return 
 */
int ieee80211_recv_radio_measurement_req(wlan_if_t vap, wlan_node_t ni,
    struct ieee80211_action_rm_req *rm_req , int  frm_len)
{
    struct ieee80211_measreq_ie *req;
    ieee80211_rrm_t rrm = vap->rrm;

    RRM_FUNCTION_ENTER;

    req = (struct ieee80211_measreq_ie *)(&(rm_req->req_ies[0]));
    rrm->ni = ni;
    if(rrm->rrm_in_progress) {
        rrm_send_reject(rrm,rm_req);
        return EOK;
    }
    else
        rrm->rrm_in_progress = 1;

    switch(req->reqtype)
    {
        case IEEE80211_MEASREQ_BASIC_REQ:
        case IEEE80211_MEASREQ_CCA_REQ:
        case IEEE80211_MEASREQ_RPI_HISTOGRAM_REQ:
        case IEEE80211_MEASREQ_TSMREQ_TYPE:
        case IEEE80211_MEASREQ_FRAME_REQ:
            break;
        case IEEE80211_MEASREQ_CHANNEL_LOAD_REQ:
            ieee80211_rrm_recv_chload_req(vap,ni,req,rm_req->dialogtoken);
            break;
        case IEEE80211_MEASREQ_NOISE_HISTOGRAM_REQ:
            ieee80211_rrm_recv_nhist_req(vap,ni,req,rm_req->dialogtoken);
            break;
        case IEEE80211_MEASREQ_BR_TYPE:
            ieee80211_rrm_recv_beacon_req(vap,ni,req,rm_req->dialogtoken);
            break;
        case IEEE80211_MEASREQ_STA_STATS_REQ:
            ieee80211_rrm_recv_stastats_req(vap,ni,req,rm_req->dialogtoken); 
            break;
        case IEEE80211_MEASREQ_LCI_REQ:
            ieee80211_rrm_recv_lci_req(vap,ni,req,rm_req->dialogtoken);
            break;
        default:
            break;
    }

    RRM_FUNCTION_EXIT;
    return EOK;
}
#endif
